<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Part of mod_plugnmeet.
 *
 * @package     mod_plugnmeet
 * @author     Jibon L. Costa <jibon@mynaparrot.com>
 * @copyright  2022 MynaParrot
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class PlugNmeetHelper {
    /**
     * @param $items
     * @param $fieldname
     * @param $data
     * @param MoodleQuickForm $mform
     * @return void
     * @throws coding_exception
     */
    private static function format_html($items, $fieldname, $data, $mform) {
        foreach ($items as $key => $item) {
            // If 'type' is not set, we assume it's a nested array of fields.
            if (empty($item['type'])) {
                $newfieldname = $fieldname . '[' . $key . ']';
                $newdata = isset($data[$key]) && is_array($data[$key]) ? $data[$key] : [];
                $mform->addElement('html', '<hr />');
                self::format_html($item, $newfieldname, $newdata, $mform);
                continue;
            }

            $elementname = "{$fieldname}[{$key}]";

            if (isset($item["disable_if"])) {
                $mform->disabledIf($elementname, $item["disable_if"]["depend_on"],
                    $item["disable_if"]["condition"], $item["disable_if"]["value"]);
            }

            if ($item["type"] === "select") {
                $select = $mform->addElement(
                    'select',
                    $elementname,
                    $item['label'],
                    $item["options"]
                );

                $value = $item["selected"];
                if (isset($data[$key])) {
                    $value = $data[$key];
                }

                foreach (array_keys($item["options"]) as $k) {
                    if ($k == $value) {
                        $select->setSelected($k);
                    }
                }
            } else if ($item["type"] === "text" || $item["type"] === "number") {
                $value = $item["default"];
                if (isset($data[$key])) {
                    $value = $data[$key];
                }
                $mform->addElement("text", $elementname, $item['label']);
                $mform->setDefault($elementname, $value);

                if ($item["type"] === "text") {
                    $mform->setType($elementname, PARAM_NOTAGS);
                } else if ($item["type"] === "number") {
                    $mform->setType($elementname, PARAM_INT);
                }
            } else if ($item["type"] === "textarea") {
                $value = $item["default"];
                if (isset($data[$key])) {
                    $value = $data[$key];
                }
                $mform->addElement(
                    'textarea',
                    $elementname,
                    $item['label'],
                    'wrap="virtual" rows="5" cols="50"'
                );
                $mform->setDefault($elementname, $value);
                $mform->setType($elementname, PARAM_CLEANHTML);
            }
        }
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_room_features($roommetadata, $mform) {
        $roomfeatures = array(
            "allow_webcams" => array(
                "label" => get_string("allow_webcams", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "mute_on_start" => array(
                "label" => get_string("mute_on_start", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "allow_screen_share" => array(
                "label" => get_string("allow_screen_share", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allow_rtmp" => array(
                "label" => get_string("allow_rtmp", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allow_view_other_webcams" => array(
                "label" => get_string("allow_view_other_webcams", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allow_view_other_users_list" => array(
                "label" => get_string("allow_view_other_users_list", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "admin_only_webcams" => array(
                "label" => get_string("admin_only_webcams", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "room_duration" => array(
                "label" => get_string("room_duration", "mod_plugnmeet"),
                "default" => 0,
                "type" => "number"
            ),
            "moderator_join_first" => array(
                "label" => get_string("moderator_join_first", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "enable_analytics" => array(
                "label" => get_string("enable_analytics", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allow_virtual_bg" => array(
                "label" => get_string("allow_virtual_bg", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allow_raise_hand" => array(
                "label" => get_string("allow_raise_hand", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "auto_gen_user_id" => array(
                "label" => get_string("auto_gen_user_id", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
        );

        $data = [];
        if (isset($roommetadata['room_features'])) {
            $data = $roommetadata['room_features'];
        }

        self::format_html($roomfeatures, "room_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_recording_features($roommetadata, $mform) {
        $recordingfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "is_allow_cloud" => array(
                "label" => get_string("allow_cloud_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "recording_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "enable_auto_cloud_recording" => array(
                "label" => get_string("enable_auto_cloud_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "recording_features[is_allow_cloud]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "is_allow_local" => array(
                "label" => get_string("allow_local_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "recording_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "is_allow_view_recording" => array(
                "label" => get_string("is_allow_view_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "recording_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "is_allow_download_recording" => array(
                "label" => get_string("is_allow_download_recording", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "recording_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
        );

        $data = [];
        if (isset($roommetadata["recording_features"])) {
            $data = $roommetadata["recording_features"];
        }

        self::format_html($recordingfeatures, "recording_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_chat_features($roommetadata, $mform) {
        $chatfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_chat", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "is_allow_file_upload" => array(
                "label" => get_string("allow_file_upload", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "chat_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
        );

        $data = [];
        if (isset($roommetadata["chat_features"])) {
            $data = $roommetadata["chat_features"];
        }

        self::format_html($chatfeatures, "chat_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_shared_note_pad_features($roommetadata, $mform) {
        $sharednotepadfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_shared_notepad", "plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            )
        );

        $data = [];
        if (isset($roommetadata["shared_note_pad_features"])) {
            $data = $roommetadata["shared_note_pad_features"];
        }

        self::format_html($sharednotepadfeatures, "shared_note_pad_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_whiteboard_features($roommetadata, $mform) {
        $whiteboardfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_whiteboard", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            )
        );

        $data = [];
        if (isset($roommetadata["whiteboard_features"])) {
            $data = $roommetadata["whiteboard_features"];
        }

        self::format_html($whiteboardfeatures, "whiteboard_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_external_media_player_features($roommetadata, $mform) {
        $externalmediaplayerfeatures = array(
            "is_allow" => array(
                "label" => get_string("allowed_external_media_player", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            )
        );

        $data = [];
        if (isset($roommetadata["external_media_player_features"])) {
            $data = $roommetadata["external_media_player_features"];
        }

        self::format_html($externalmediaplayerfeatures, "external_media_player_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_waiting_room_features($roommetadata, $mform) {
        $waitingroomfeatures = array(
            "is_active" => array(
                "label" => get_string("activate_waiting_room", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "waiting_room_msg" => array(
                "label" => get_string("waiting_room_msg", "mod_plugnmeet"),
                "default" => "",
                "type" => "textarea",
                "disable_if" => array(
                    "depend_on" => "waiting_room_features[is_active]",
                    "condition" => "eq",
                    "value" => "0"
                )
            )
        );

        $data = [];
        if (isset($roommetadata["waiting_room_features"])) {
            $data = $roommetadata["waiting_room_features"];
        }

        self::format_html($waitingroomfeatures, "waiting_room_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_breakout_room_features($roommetadata, $mform) {
        $breakoutroomfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_breakout_rooms", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "allowed_number_rooms" => array(
                "label" => get_string("allowed_number_rooms", "mod_plugnmeet"),
                "default" => 6,
                "type" => "number",
                "disable_if" => array(
                    "depend_on" => "breakout_room_features[is_allow]",
                    "condition" => "eq",
                    "value" => "0"
                )
            )
        );

        $data = [];
        if (isset($roommetadata["breakout_room_features"])) {
            $data = $roommetadata["breakout_room_features"];
        }

        self::format_html($breakoutroomfeatures, "breakout_room_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_display_external_link_features($roommetadata, $mform) {
        $displayexternallinkfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_display_external_link_features", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
        );

        $data = [];
        if (isset($roommetadata["display_external_link_features"])) {
            $data = $roommetadata["display_external_link_features"];
        }

        self::format_html($displayexternallinkfeatures, "display_external_link_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_ingress_features($roommetadata, $mform) {
        $ingressfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_ingress_features", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
        );

        $data = [];
        if (isset($roommetadata["ingress_features"])) {
            $data = $roommetadata["ingress_features"];
        }

        self::format_html($ingressfeatures, "ingress_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_polls_features($roommetadata, $mform) {
        $pollsfeatures = array(
            "is_allow" => array(
                "label" => get_string("allow_polls", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
        );

        $data = [];
        if (isset($roommetadata["polls_features"])) {
            $data = $roommetadata["polls_features"];
        }

        self::format_html($pollsfeatures, "polls_features", $data, $mform);
    }

    /*
    * @param $roommetadata
    * @param $mform
    * @return void
    * @throws coding_exception
    */
    public static function get_end_to_end_encryption_features($roommetadata, $mform) {
        $e2eefeatures = array(
            "is_enabled" => array(
                "label" => get_string("enable_end_to_end_encryption_features", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "enabled_self_insert_encryption_key" => array(
                "label" => get_string("enabled_self_insert_encryption_key", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "end_to_end_encryption_features[is_enabled]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "included_chat_messages" => array(
                "label" => get_string("included_e2ee_chat_messages", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "end_to_end_encryption_features[is_enabled]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
            "included_whiteboard" => array(
                "label" => get_string("included_e2ee_whiteboard", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select",
                "disable_if" => array(
                    "depend_on" => "end_to_end_encryption_features[is_enabled]",
                    "condition" => "eq",
                    "value" => "0"
                )
            ),
        );

        $data = [];
        if (isset($roommetadata["end_to_end_encryption_features"])) {
            $data = $roommetadata["end_to_end_encryption_features"];
        }

        self::format_html($e2eefeatures, "end_to_end_encryption_features", $data, $mform);
    }

    /*
    * @param $roommetadata
    * @param $mform
    * @return void
    * @throws coding_exception
    */
    public static function get_insights_features($roommetadata, $mform) {
        $insightsfeatures = array(
            "is_allow" => array(
                "label" => get_string("insights_features_is_allow", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "transcription_features" => array(
                "is_allow" => array(
                    "label" => get_string("insights_transcription_features_is_allow", "mod_plugnmeet"),
                    "options" => array(
                        0 => get_string("no", "mod_plugnmeet"),
                        1 => get_string("yes", "mod_plugnmeet")
                    ),
                    "selected" => 0,
                    "type" => "select",
                    "disable_if" => array(
                        "depend_on" => "insights_features[is_allow]",
                        "condition" => "eq",
                        "value" => "0"
                    )
                ),
                "is_allow_translation" => array(
                    "label" => get_string("insights_transcription_features_is_allow_translation", "mod_plugnmeet"),
                    "options" => array(
                        0 => get_string("no", "mod_plugnmeet"),
                        1 => get_string("yes", "mod_plugnmeet")
                    ),
                    "selected" => 0,
                    "type" => "select",
                    "disable_if" => array(
                        "depend_on" => "insights_features[transcription_features][is_allow]",
                        "condition" => "eq",
                        "value" => "0"
                    )
                ),
                "is_allow_speech_synthesis" => array(
                    "label" => get_string("insights_transcription_features_is_allow_speech_synthesis", "mod_plugnmeet"),
                    "options" => array(
                        0 => get_string("no", "mod_plugnmeet"),
                        1 => get_string("yes", "mod_plugnmeet")
                    ),
                    "selected" => 0,
                    "type" => "select",
                    "disable_if" => array(
                        "depend_on" => "insights_features[transcription_features][is_allow]",
                        "condition" => "eq",
                        "value" => "0"
                    )
                ),
            ),
            "chat_translation_features" => array(
                "is_allow" => array(
                    "label" => get_string("insights_chat_translation_features_is_allow", "mod_plugnmeet"),
                    "options" => array(
                        0 => get_string("no", "mod_plugnmeet"),
                        1 => get_string("yes", "mod_plugnmeet")
                    ),
                    "selected" => 0,
                    "type" => "select",
                    "disable_if" => array(
                        "depend_on" => "insights_features[is_allow]",
                        "condition" => "eq",
                        "value" => "0"
                    )
                ),
            ),
            "ai_features" => array(
                "is_allow" => array(
                    "label" => get_string("insights_ai_features_is_allow", "mod_plugnmeet"),
                    "options" => array(
                        0 => get_string("no", "mod_plugnmeet"),
                        1 => get_string("yes", "mod_plugnmeet")
                    ),
                    "selected" => 0,
                    "type" => "select",
                    "disable_if" => array(
                        "depend_on" => "insights_features[is_allow]",
                        "condition" => "eq",
                        "value" => "0"
                    )
                ),
                "ai_text_chat_features" => array(
                    "is_allow" => array(
                        "label" => get_string("insights_ai_text_chat_features_is_allow", "mod_plugnmeet"),
                        "options" => array(
                            0 => get_string("no", "mod_plugnmeet"),
                            1 => get_string("yes", "mod_plugnmeet")
                        ),
                        "selected" => 0,
                        "type" => "select",
                        "disable_if" => array(
                            "depend_on" => "insights_features[ai_features][is_allow]",
                            "condition" => "eq",
                            "value" => "0"
                        )
                    ),
                ),
                "meeting_summarization_features" => array(
                    "is_allow" => array(
                        "label" => get_string("insights_ai_meeting_summarization_features_is_allow", "mod_plugnmeet"),
                        "options" => array(
                            0 => get_string("no", "mod_plugnmeet"),
                            1 => get_string("yes", "mod_plugnmeet")
                        ),
                        "selected" => 0,
                        "type" => "select",
                        "disable_if" => array(
                            "depend_on" => "insights_features[ai_features][is_allow]",
                            "condition" => "eq",
                            "value" => "0"
                        )
                    ),
                )
            )
        );

        $data = [];
        if (isset($roommetadata["insights_features"])) {
            $data = $roommetadata["insights_features"];
        }

        self::format_html($insightsfeatures, "insights_features", $data, $mform);
    }

    /**
     * @param $roommetadata
     * @param $mform
     * @return void
     * @throws coding_exception
     */
    public static function get_default_lock_settings($roommetadata, $mform) {
        $defaultlocksettings = array(
            "lock_microphone" => array(
                "label" => get_string("lock_microphone", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "lock_webcam" => array(
                "label" => get_string("lock_webcam", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "lock_screen_sharing" => array(
                "label" => get_string("lock_screen_sharing", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "lock_whiteboard" => array(
                "label" => get_string("lock_whiteboard", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "lock_shared_notepad" => array(
                "label" => get_string("lock_shared_notepad", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 1,
                "type" => "select"
            ),
            "lock_chat" => array(
                "label" => get_string("lock_chat", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "lock_chat_send_message" => array(
                "label" => get_string("lock_chat_send_message", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "lock_chat_file_share" => array(
                "label" => get_string("lock_chat_file_share", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
            "lock_private_chat" => array(
                "label" => get_string("lock_private_chat", "mod_plugnmeet"),
                "options" => array(
                    0 => get_string("no", "mod_plugnmeet"),
                    1 => get_string("yes", "mod_plugnmeet")
                ),
                "selected" => 0,
                "type" => "select"
            ),
        );

        $data = [];
        if (isset($roommetadata["default_lock_settings"])) {
            $data = $roommetadata["default_lock_settings"];
        }

        self::format_html($defaultlocksettings, "default_lock_settings", $data, $mform);
    }

    /**
     * @param $id
     * @return void
     */
    public static function show_join_button($id) {
        echo "<a href='/mod/plugnmeet/conference.php?id={$id}' class='btn btn-success'>" . get_string("join", "mod_plugnmeet") . "</a>";
    }
}
