// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Handle the configuration of the roadmap.
 *
 * @module     mod_roadmap/roadmapconfig
 * @copyright  2021 Steve Bader <smbader@ncsu.edu>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define([
    'jquery',
    'core/notification',
    'core/templates',
    'core/str',
    'mod_roadmap/expand_contract',
    'mod_roadmap/learningobjectivesconfig',
    'mod_roadmap/step_icon_select',
    'mod_roadmap/step_activity_select',
    'mod_roadmap/step_save',
    'mod_roadmap/cycle_save',
    'mod_roadmap/phase_save',
    'mod_roadmap/repository',
    'core/modal_delete_cancel',
    'core/modal_events'
], (
    $,
    notification,
    templates,
    Str,
    expandContract,
    learningObjectives,
    stepIconSelect,
    stepActivitySelect,
    stepSave,
    cycleSave,
    phaseSave,
    roadmapRepo,
    ModalDeleteCancel,
    ModalEvents
) => {

    const showDeleteConfirmModal = (message, title = Str.get_string('confirmdelete', 'mod_roadmap')) => {
        return ModalDeleteCancel.create({
            title,
            body: message
        }).then(modal => {
            return new Promise(resolve => {
                modal.show();
                modal.getRoot().on(ModalEvents.delete, () => {
                    resolve(true);
                    modal.hide();
                });
                modal.getRoot().on(ModalEvents.cancel, () => {
                    resolve(false);
                    modal.hide();
                });
                modal.getRoot().on(ModalEvents.hidden, () => {
                    modal.destroy();
                });
            });
        });
    };

    const getMaxValue = dataType => {
        const ids = [];
        $(`.${dataType}-wrapper`).each(function() {
            const id = parseInt($(this).data(`${dataType}id`), 10);
            if (!isNaN(id)) {
                ids.push(id);
            }
        });
        return ids.length ? Math.max(...ids) : 0;
    };

    const moveWrapper = (wrapper, direction) => {
        if (direction === 'up') {
            const prev = wrapper.prev();
            if (prev.length) {
                wrapper.insertBefore(prev);
            }
        } else if (direction === 'down') {
            const next = wrapper.next();
            if (next.length) {
                wrapper.insertAfter(next);
            }
        }
    };

    let colorPatternCache = {};

    class RoadmapConfig {
        constructor(inputSelector, inputConfig) {
            this.inputSelector = inputSelector;
            this.configContainer = $(inputConfig);
            this.initForm();
        }

        initForm() {
            let inputConfigVal = this.configContainer.val();
            if (!inputConfigVal) {
                inputConfigVal = JSON.stringify({phases: []});
                this.configContainer.val(inputConfigVal);
            }
            let config;
            try {
                config = JSON.parse(inputConfigVal);
            } catch (e) {
                notification.exception(e);
                config = {phases: []};
            }

            config.phases.forEach(phase => {
                phase.configuration = JSON.stringify(phase);
                (phase.cycles || []).forEach(cycle => {
                    cycle.configuration = JSON.stringify(cycle);
                    (cycle.steps || []).forEach(step => {
                        step.configuration = JSON.stringify(step);
                    });
                });
            });

            templates.render('mod_roadmap/configuration_phases', config)
                .then((html, js) => {
                    templates.prependNodeContents(this.inputSelector, html, js);

                    this.bindExpandCollapseControls();

                    $('#roadmapconfiguration')
                        .off('click', 'a[data-action]')
                        .on('click', 'a[data-action]', e => this.clickHandler(e));

                    this.phaseColorChange($('select[name="phasecolorpattern"]'));

                    learningObjectives.refreshChecklists();
                    stepIconSelect.rebindButtons();
                    stepActivitySelect.rebindButtons();
                    stepSave.rebindInputs();
                    cycleSave.rebindInputs();
                    phaseSave.rebindInputs();
                    this.bindConfigSave();

                    require(['theme_boost/loader']);
                    return null;
                })
                .catch(notification.exception);

            $('select[name="phasecolorpattern"]').off('change').on('change', e => {
                this.phaseColorChange($(e.target));
            });
        }

        clickHandler(event) {
            const node = $(event.currentTarget);
            const action = node.data('action');
            const actionMap = {
                'collapse_all_phases': () => this.collapsePhases(),
                'expand_all_phases': () => this.expandPhases(),
                'collapse_all_cycles': () =>
                    this.collapseCycles(
                        node.closest('.phase-wrapper')
                            .find('.phase-cycles-container')
                            .first()
                    ),
                'expand_all_cycles': () =>
                    this.expandCycles(
                        node.closest('.phase-wrapper')
                            .find('.phase-cycles-container')
                            .first()
                    ),
                'collapse_all_steps': () =>
                    this.collapseSteps(
                        node.parent('.step-container-controls')
                            .next('.cycle-steps-container')
                    ),
                'expand_all_steps': () =>
                    this.expandSteps(
                        node.parent('.step-container-controls')
                            .next('.cycle-steps-container')
                    ),
                'phase_collapse_control': () => this.collapsePhase(node),
                'cycle_collapse_control': () => this.collapseCycle(node),
                'step_collapse_control': () => this.collapseStep(node),
                'phase_delete_control': () => this.deletePhase(node),
                'cycle_delete_control': () => this.deleteCycle(node),
                'step_delete_control': () => this.deleteStep(node),
                'add_phase': () => this.addPhase(),
                'phase_up_control': () => this.upPhase(node),
                'phase_down_control': () => this.downPhase(node),
                'add_phase_cycle': () => this.addCycle(node),
                'cycle_up_control': () => this.upCycle(node),
                'cycle_down_control': () => this.downCycle(node),
                'add_cycle_step': () => this.addStep(node),
                'step_up_control': () => this.upStep(node),
                'step_down_control': () => this.downStep(node)
            };
            if (actionMap[action]) {
                return actionMap[action]();
            }
            return undefined;
        }

        phaseColorChange(node) {
            const colorId = node.val();
            if (colorPatternCache[colorId]) {
                this.applyColorSet(colorPatternCache[colorId]);
                return;
            }
            roadmapRepo.fetchColorPattern(colorId)
                .then(colors => {
                    colorPatternCache[colorId] = colors;
                    this.applyColorSet(colors);
                    return colors;
                })
                .catch(notification.exception);
        }

        applyColorSet(colors) {
            $('.phase-color-display').remove();
            const colorTable = $('<div>').addClass('phase-color-display');
            colors.forEach(color => {
                colorTable.append(`<div class="color" style="background-color:${color}"></div>`);
            });
            $('select[name="phasecolorpattern"]').parent().append(colorTable);

            $('#roadmapconfiguration #phase-container > .phase-wrapper > div.row').each(function(i) {
                const numColors = colors.length;
                const idxColor = i % numColors;
                $(this).css('border-bottom', `solid 1px ${colors[idxColor]}`);
            });
        }

        bindConfigSave() {
            $('input.phase-configuration').off('change').on('change', () => this.configSave());
        }

        configSave() {
            const phaseContainer = $('#phase-container');
            const roadmapData = [];
            let index = 0;
            phaseContainer.find('.phase-wrapper .phase-configuration').each(function() {
                const phaseData = $(this).val() || '{}';
                const phaseDataObj = JSON.parse(phaseData);
                phaseDataObj.index = index++;
                roadmapData.push(phaseDataObj);
            });
            $('input[name="roadmapconfiguration"]').val(JSON.stringify({
                phases: roadmapData,
                phaseDeletes: $('#phase-deletes').val(),
                cycleDeletes: $('#cycle-deletes').val(),
                stepDeletes: $('#step-deletes').val()
            }));
        }

        addPhase() {
            const roadmapConfigInput = $('input[name="roadmapconfiguration"]');
            const config = JSON.parse(roadmapConfigInput.val());
            const nextIndex = config.phases.length;
            const maxPhaseId = getMaxValue('phase');
            const newPhase = {
                id: maxPhaseId + 1,
                index: nextIndex,
                number: nextIndex + 1,
                title: Str.get_string('phase', 'mod_roadmap') + `${nextIndex + 1}`,
                subtitle: Str.get_string('subtitle', 'mod_roadmap') + `${nextIndex + 1}`
            };
            config.phases.push(newPhase);
            roadmapConfigInput.val(JSON.stringify(config));
            return templates.render('mod_roadmap/configuration_phase', newPhase)
                .then((html, js) => {
                    templates.appendNodeContents('#phase-container', html, js);
                    phaseSave.rebindInputs();
                    this.bindConfigSave();
                    this.phaseColorChange($('select[name="phasecolorpattern"]'));
                    return null;
                })
                .catch(notification.exception);
        }

        addCycle(node) {
            const phaseContainer = node.closest('.phase-container');
            const cycleContainer = phaseContainer.children('.phase-cycles-container');
            const nextCycleIndex = cycleContainer.children('.cycle-wrapper').length;
            const maxCycleId = getMaxValue('cycle');
            const newCycle = {
                id: maxCycleId + 1,
                index: nextCycleIndex,
                number: nextCycleIndex + 1,
                title: Str.get_string('cycle', 'mod_roadmap') + `${nextCycleIndex + 1}`,
                subtitle: Str.get_string('subtitle', 'mod_roadmap') + `${nextCycleIndex + 1}`
            };
            return templates.render('mod_roadmap/configuration_cycle', newCycle)
                .then((html, js) => {
                    templates.appendNodeContents(cycleContainer, html, js);
                    learningObjectives.refreshChecklists();
                    cycleSave.rebindInputs();
                    phaseSave.rebindInputs();
                    return null;
                })
                .catch(notification.exception);
        }

        addStep(node) {
            const cycleContainer = node.closest('.cycle-container');
            const stepsContainer = cycleContainer.children('.cycle-steps-container');
            const nextStepIndex = stepsContainer.children('.step-wrapper').length;
            const maxStepId = getMaxValue('step');
            const iconUrl = $('input[name="icon_url"]').val();

            const newStep = {
                id: maxStepId + 1,
                index: nextStepIndex,
                number: nextStepIndex + 1,
                stepicon: 'icon-10',
                iconurl: `${iconUrl}?name=icon-10&percent=100&flags=n`
            };
            return templates.render('mod_roadmap/configuration_step', newStep)
                .then((html, js) => {
                    templates.appendNodeContents(stepsContainer, html, js);
                    stepIconSelect.rebindButtons();
                    stepActivitySelect.rebindButtons();
                    stepSave.rebindInputs();
                    cycleSave.rebindInputs();
                    return null;
                })
                .catch(notification.exception);
        }

        bindExpandCollapseControls() {
            $('.expand-collapse-controls .collapse_everything').off('click').on('click', e => {
                e.preventDefault();
                this.collapseEverything();
            });
            $('.expand-collapse-controls .expand_everything').off('click').on('click', e => {
                e.preventDefault();
                this.expandEverything();
            });
        }

        expandPhases() {
            this.expandChildrenClass($('#roadmapconfiguration'), 'phase');
        }
        expandCycles(node) {
            this.expandChildrenClass(node, 'cycle');
        }
        expandSteps(node) {
            this.expandChildrenClass(node, 'step');
        }
        collapsePhases() {
            const node = $('#roadmapconfiguration');
            this.collapseCycles(node);
            this.collapseChildrenClass(node, 'phase');
        }
        collapseCycles(node) {
            this.collapseSteps(node);
            this.collapseChildrenClass(node, 'cycle');
        }
        collapseSteps(node) {
            this.collapseChildrenClass(node, 'step');
        }
        expandChildrenClass(parent, className) {
            parent.find(`a[data-action="${className}_collapse_control"]`).each(function() {
                const wrapper = $(this).closest(`.${className}-wrapper`);
                const container = wrapper.children(`.${className}-container`);
                if (container.hasClass('hide')) {
                    $(this).click();
                }
            });
        }
        collapseChildrenClass(parent, className) {
            parent.find(`a[data-action="${className}_collapse_control"]`).each(function() {
                const wrapper = $(this).closest(`.${className}-wrapper`);
                const container = wrapper.children(`.${className}-container`);
                if (container.hasClass('visible')) {
                    $(this).click();
                }
            });
        }

        collapsePhase(node) {
            const wrapper = node.closest('.phase-wrapper');
            const container = wrapper.children('.phase-container');
            expandContract.expandCollapse(container, node);
        }
        collapseCycle(node) {
            const wrapper = node.closest('.cycle-wrapper');
            const container = wrapper.children('.cycle-container');
            expandContract.expandCollapse(container, node);
        }
        collapseStep(node) {
            const wrapper = node.closest('.step-wrapper');
            const container = wrapper.children('.step-container');
            expandContract.expandCollapse(container, node);
        }

        deletePhase(node) {
            return showDeleteConfirmModal(
                Str.get_string('confirmdeletephase', 'mod_roadmap'),
                Str.get_string('confirmdelete', 'mod_roadmap')
            ).then(confirmed => {
                if (confirmed) {
                    const wrapper = node.closest('.phase-wrapper');
                    if (!wrapper.length) {
                        return null;
                    }
                    const phaseId = wrapper.data('phaseid');
                    const phaseDeletes = $('#phase-deletes');
                    phaseDeletes.val(phaseDeletes.val() + phaseId + ',');
                    wrapper.remove();
                    this.configSave();
                    this.phaseColorChange($('select[name="phasecolorpattern"]'));
                }
                return null;
            });
        }

        deleteCycle(node) {
            return showDeleteConfirmModal(
                Str.get_string('confirmdeletecycle', 'mod_roadmap'),
                Str.get_string('confirmdelete', 'mod_roadmap')
            ).then(confirmed => {
                if (confirmed) {
                    const wrapper = node.closest('.cycle-wrapper');
                    if (!wrapper.length) {
                        return null;
                    }
                    const phaseContainer = node.closest('.phase-container');
                    const cycleId = wrapper.data('cycleid');
                    const cycleDeletes = $('#cycle-deletes');
                    cycleDeletes.val(cycleDeletes.val() + cycleId + ',');
                    wrapper.remove();
                    phaseContainer.find('.phase-title').triggerHandler('change');
                }
                return null;
            });
        }

        deleteStep(node) {
            return showDeleteConfirmModal(
                Str.get_string('confirmdeletestep', 'mod_roadmap'),
                Str.get_string('confirmdelete', 'mod_roadmap')
            ).then(confirmed => {
                if (confirmed) {
                    const wrapper = node.closest('.step-wrapper');
                    if (!wrapper.length) {
                        return null;
                    }
                    const cycleContainer = node.closest('.cycle-container');
                    const stepId = wrapper.data('stepid');
                    const stepDeletes = $('#step-deletes');
                    stepDeletes.val(stepDeletes.val() + stepId + ',');
                    wrapper.remove();
                    cycleContainer.find('.cycle-title').triggerHandler('change');
                }
                return null;
            });
        }

        upPhase(node) {
            const wrapper = node.closest('.phase-wrapper');
            moveWrapper(wrapper, 'up');
            this.configSave();
            this.phaseColorChange($('select[name="phasecolorpattern"]'));
        }
        downPhase(node) {
            const wrapper = node.closest('.phase-wrapper');
            moveWrapper(wrapper, 'down');
            this.configSave();
            this.phaseColorChange($('select[name="phasecolorpattern"]'));
        }
        upCycle(node) {
            const wrapper = node.closest('.cycle-wrapper');
            const phaseContainer = node.closest('.phase-container');
            moveWrapper(wrapper, 'up');
            phaseContainer.find('.phase-title').triggerHandler('change');
        }
        downCycle(node) {
            const wrapper = node.closest('.cycle-wrapper');
            const phaseContainer = node.closest('.phase-container');
            moveWrapper(wrapper, 'down');
            phaseContainer.find('.phase-title').triggerHandler('change');
        }
        upStep(node) {
            const wrapper = node.closest('.step-wrapper');
            const cycleContainer = node.closest('.cycle-container');
            moveWrapper(wrapper, 'up');
            cycleContainer.find('.cycle-title').triggerHandler('change');
        }
        downStep(node) {
            const wrapper = node.closest('.step-wrapper');
            const cycleContainer = node.closest('.cycle-container');
            moveWrapper(wrapper, 'down');
            cycleContainer.find('.cycle-title').triggerHandler('change');
        }

        expandEverything() {
            this.expandPhases();
        }
        collapseEverything() {
            this.collapsePhases();
        }
    }

    return {
        init: (inputSelector, inputConfig) => new RoadmapConfig(inputSelector, inputConfig)
    };
});