<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Badge test
 *
 * @package    local_obf
 * @copyright  2013-2020, Open Badge Factory Oy
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once(__DIR__ . '/../classes/badge.php');
require_once(__DIR__ . '/../classes/client.php');

/**
 * Badge testcase.
 *
 * @group obf
 * @copyright  2013-2020, Open Badge Factory Oy
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

class local_obf_badge_testcase extends advanced_testcase {
    /**
     * Simple provider of badge data.
     */
    public function badgeprovider() {
        return array(
            array(
                array(
                    array(
                        "ctime" => "1383658770",
                        "mtime" => "1383658770",
                        "name" => "Inspector Gadget",
                        "readyforissuing" => true,
                        "client_id" => "MTKUR18NL1",
                        "description" => "Inspected the undocumented code of Moodle 2.2",
                        "image" => "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAYAAACtWK6eAAAgAElEQVR4Xu1dB3xURRr/NqRTklAE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\ngg==\n",
                        "draft" => "0",
                        "id" => "MVSLXUiDGCi1",
                        "criteria" => "https://elvis.discendum.com/obf/v1/badge/_/MVSLXUiDGCi1/criteria.html",
                        "expires" => "6"),
                    array(
                        "ctime" => "1383049193",
                        "mtime" => "1383049193",
                        "name" => "Sleepy Joe",
                        "client_id" => "MTKUR18NL1",
                        "description" => "You're a decent sleeper",
                        "image" => "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAYAAACtWK6eAAAgAElEQVR4Xu1dB5hURda9Q85JkkoY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\nAABJRU5ErkJggg==\n",
                        "draft" => "1",
                        "id" => "MVFJL5wET2w1",
                        "readyforissuing" => true,
                        "criteria" => "https://elvis.discendum.com/obf/v1/badge/_/MVFJL5wET2w1/criteria.html",
                        "expires" => "0"
                    )
                )
            )
        );
    }

    /**
     * Test badges.
     *
     * @param array $badgearr
     * @dataProvider badgeprovider
     */
    public function test_get_badges($badgearr) {
        $this->resetAfterTest();

        $stub = $this->getMockBuilder(\classes\obf_client::class)
            ->onlyMethods(['get_badges'])
            ->getMock();
        $stub->expects($this->any())->method('get_badges')->will($this->returnValue($badgearr));

        $badges = \classes\obf_badge::get_badges($stub);

        $this->assertCount(2, $badges);
        $this->assertEquals('Sleepy Joe', $badges['MVFJL5wET2w1']->get_name());
        $this->assertTrue($badges['MVFJL5wET2w1']->is_draft());
        $this->assertFalse($badges['MVSLXUiDGCi1']->is_draft());
        $this->assertFalse($badges['MVFJL5wET2w1']->has_expiration_date());

        // Check month difference, cannot do comparison with seconds because of
        // possible inaccuracies.
        $d1 = new DateTime();
        $d2 = new DateTime(date('Y-m-d', $badges['MVSLXUiDGCi1']->get_expires()));
        $months = $d2->diff($d1)->m;
        $this->assertTrue($months == 6 || $months == 5);
    }
}
