<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace block_course_recommender;

defined('MOODLE_INTERNAL') || die();

require_once("$CFG->libdir/externallib.php");

use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core\context\system as context_system;
use moodle_url;
use moodle_exception;

/**
 * Course recommender external API
 *
 * @package    block_course_recommender
 * @copyright  2025 Sadik Mert
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class external extends external_api {
    /**
     * Returns description of get_courses parameters
     *
     * @return external_function_parameters
     */
    public static function get_courses_parameters() {
        return new external_function_parameters([
            'interests' => new external_multiple_structure(
                new external_value(PARAM_TEXT, 'Interest tag'),
                'List of selected interests',
                VALUE_DEFAULT,
                []
            ),
            'sesskey' => new external_value(PARAM_RAW, 'Session key for security'),
        ]);
    }

    /**
     * Returns description of get_courses result value
     *
     * @return external_single_structure
     */
    public static function get_courses_returns() {
        return new external_single_structure([
            'html' => new external_value(PARAM_RAW, 'HTML content for the courses list'),
        ]);
    }

    /**
     * Get courses based on interests
     *
     * @param array $interests List of interests
     * @param string $sesskey Session key
     * @return array
     */
    public static function get_courses($interests, $sesskey) {
        global $PAGE, $OUTPUT;

        // Context and parameters validation.
        $context = context_system::instance();
        $PAGE->set_context($context);
        self::validate_context($context);
        require_capability('moodle/course:view', $context);

        $params = self::validate_parameters(self::get_courses_parameters(), [
            'interests' => $interests,
            'sesskey' => $sesskey,
        ]);

        if (!confirm_sesskey($params['sesskey'])) {
            throw new moodle_exception('invalidsesskey', 'error');
        }

        if (empty($params['interests'])) {
            return ['html' => ''];
        }

        $tagids = self::get_tag_ids_by_rawnames($params['interests']);
        if (empty($tagids)) {
            return ['html' => '<div class="alert alert-info">' . get_string('nocourses', 'block_course_recommender') . '</div>'];
        }

        $courses = self::find_matching_courses($tagids);
        $selectedinterests = array_map('mb_strtolower', array_map('trim', $params['interests']));
        $courselist = self::prepare_course_list_data($courses, $selectedinterests);

        $data = [
            'matchingcourses' => get_string('matchingcourses', 'block_course_recommender'),
            'nocourses' => get_string('nocourses', 'block_course_recommender'),
            'tagcolor' => self::get_tagcolor(),
        ];
        if (!empty($courselist)) {
            $data['courses'] = [
                'list' => $courselist,
            ];
        }
        $html = $OUTPUT->render_from_template('block_course_recommender/courses', $data);
        return ['html' => $html];
    }

    /**
     * Get tag IDs by their raw names.
     * @param array $interests
     * @return array
     */
    protected static function get_tag_ids_by_rawnames($interests) {
        global $DB;
        $in = $DB->get_in_or_equal($interests, SQL_PARAMS_NAMED, 'tag');
        $insql = $in[0];
        $inparams = $in[1];
        $tagrecords = $DB->get_records_select('tag', "rawname $insql", $inparams);
        if (empty($tagrecords)) {
            return [];
        }
        return array_keys($tagrecords);
    }

    /**
     * Find matching courses for the tag IDs.
     * @param array $tagids
     * @return array
     */
    protected static function find_matching_courses($tagids) {
        global $DB;
        $in1 = $DB->get_in_or_equal($tagids, SQL_PARAMS_NAMED, 'tag');
        $tagidssql = $in1[0];
        $tagidparams = $in1[1];
        $in2 = $DB->get_in_or_equal($tagids, SQL_PARAMS_NAMED, 'tag2');
        $tagidssql2 = $in2[0];
        $tagidparams2 = $in2[1];
        $groupconcat = $DB->sql_group_concat('t.rawname');
        $sql = "
            WITH matching_courses AS (
                SELECT DISTINCT c.id
                FROM {course} c
                JOIN {tag_instance} ti ON ti.itemid = c.id
                WHERE ti.tagid $tagidssql
                AND ti.itemtype = 'course'
                AND ti.component = 'core'
                AND c.visible = 1
            )
            SELECT c.*, $groupconcat as tagnames,
                   COUNT(CASE WHEN t.id $tagidssql2 THEN 1 END) as matching_tags
            FROM {course} c
            JOIN matching_courses mc ON mc.id = c.id
            LEFT JOIN {tag_instance} ti ON ti.itemid = c.id
                AND ti.itemtype = 'course'
                AND ti.component = 'core'
            LEFT JOIN {tag} t ON t.id = ti.tagid
            GROUP BY c.id
            ORDER BY matching_tags DESC, c.timecreated DESC
            LIMIT 20
        ";
        $sqlparams = array_merge($tagidparams, $tagidparams2);
        return $DB->get_records_sql($sql, $sqlparams);
    }

    /**
     * Prepare course data for the template.
     * @param array $courses
     * @param array $selectedinterests
     * @return array
     */
    protected static function prepare_course_list_data($courses, $selectedinterests) {
        $list = [];
        foreach ($courses as $course) {
            $url = new \moodle_url('/course/view.php', ['id' => $course->id]);
            $title = format_string($course->fullname);
            $courseobj = new \core_course_list_element($course);
            $image = \core_course\external\course_summary_exporter::get_course_image($courseobj);
            if (empty($image)) {
                $image = 'https://picsum.photos/400/200?random=' . $course->id;
            }
            // Filter the course tags so only those selected as interests are returned.
            $tags = [];
            if (!empty($course->tagnames)) {
                $rawtags = array_map('trim', explode(',', $course->tagnames));
                foreach ($rawtags as $t) {
                    if (in_array(mb_strtolower($t), $selectedinterests, true)) {
                        $tags[] = $t;
                    }
                }
            }
            $list[] = [
                'url' => $url->out(false),
                'title' => $title,
                'image' => $image,
                'tags' => $tags,
            ];
        }
        return $list;
    }

    /**
     * Returns the tag color from the settings.
     * @return string
     */
    protected static function get_tagcolor() {
        $tagcolor = get_config('block_course_recommender', 'tagcolor');
        if (empty($tagcolor)) {
            $tagcolor = '#0f6fc5';
        }
        return $tagcolor;
    }
}
