// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * JavaScript for image fullscreen functionality.
 *
 * @module     filter_imagefullscreen/fullscreen
 * @copyright  2025 Sandip R <radadiyasandip89@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define(['jquery', 'core/notification'], function($, Notification) {
    var currentZoom = 1;
    var isDragging = false;
    var startX;
    var startY;
    var translateX = 0;
    var translateY = 0;

    /**
     * Initialize the fullscreen functionality
     */
    var init = function() {
        createModal().then(function() {
            registerEventListeners();
            registerZoomListeners();
            return;
        }).catch(Notification.exception);
    };

    /**
     * Create the fullscreen modal
     * @return {Promise}
     */
    var createModal = function() {
        if ($('#fullscreen-modal').length) {
            return $.Deferred().resolve();
        }

        var modalHtml = '<div id="fullscreen-modal">' +
            '<span class="close-modal">&times;</span>' +
            '<img class="modal-content" alt="">' +
            '<div class="image-title"></div>' +
            '<div class="zoom-controls">' +
                '<button class="zoom-out" title="Zoom Out">−</button>' +
                '<span class="zoom-level">100%</span>' +
                '<button class="zoom-in" title="Zoom In">+</button>' +
                '<button class="zoom-reset" title="Reset Zoom">↺</button>' +
            '</div>' +
            '</div>';

        $('body').append(modalHtml);
        return $.Deferred().resolve();
    };

    /**
     * Register event listeners for the fullscreen functionality
     */
    var registerEventListeners = function() {
        // Handle fullscreen icon click
        $('body').on('click', '.fullscreen-icon', function(e) {
            e.preventDefault();
            e.stopPropagation();

            var container = $(this).closest('.image-fullscreen-container');
            var img = container.find('img');
            var imgSrc = img.attr('src');
            var imgAlt = img.attr('alt') || '';

            var modal = $('#fullscreen-modal');
            modal.find('.modal-content').attr('src', imgSrc).attr('alt', imgAlt);
            if (imgAlt) {
                modal.find('.image-title').text(imgAlt);
            } else {
                modal.find('.image-title').remove();
            }
            modal.addClass('show');

            // Enable keyboard navigation
            $(document).on('keydown.fullscreen', function(event) {
                if (event.key === 'Escape') {
                    closeModal();
                }
            });
        });

        // Close modal when clicking the X or outside the image
        $('#fullscreen-modal').on('click', function(e) {
            // Don't close if clicking on zoom controls, modal content, or any zoom control buttons
            if ($(e.target).closest('.zoom-controls').length ||
                $(e.target).is('.modal-content')) {
                return;
            }
            closeModal();
        });

        // Stop propagation for zoom control clicks
        $('.zoom-controls button').on('click', function(e) {
            e.stopPropagation();
        });
    };

    /**
     * Close the fullscreen modal
     */
    var closeModal = function() {
        var modal = $('#fullscreen-modal');
        modal.removeClass('show');
        $(document).off('keydown.fullscreen');
        resetZoom();
    };

    /**
     * Register zoom and drag related event listeners
     */
    var registerZoomListeners = function() {
        var modal = $('#fullscreen-modal');
        var modalContent = modal.find('.modal-content');

        // Zoom controls
        modal.find('.zoom-in').on('click', function() { zoom(0.1); });
        modal.find('.zoom-out').on('click', function() { zoom(-0.1); });
        modal.find('.zoom-reset').on('click', resetZoom);

        // Mouse wheel zoom
        modal.on('wheel', function(e) {
            e.preventDefault();
            zoom(e.originalEvent.deltaY < 0 ? 0.1 : -0.1);
        });

        // Drag functionality
        modalContent.on('mousedown touchstart', startDrag);
        $(document).on('mousemove touchmove', drag);
        $(document).on('mouseup touchend mouseleave', stopDrag);

        // Prevent default image drag
        modalContent.on('dragstart', function(e) { e.preventDefault(); });
    };

    /**
     * Handle zoom functionality
     * @param {number} delta - Amount to zoom in/out
     */
    var zoom = function(delta) {
        var newZoom = Math.min(Math.max(currentZoom + delta, 0.5), 3);
        if (newZoom !== currentZoom) {
            currentZoom = newZoom;
            updateTransform();
            $('#fullscreen-modal .zoom-level').text(Math.round(currentZoom * 100) + '%');
        }
    };

    /**
     * Reset zoom and position
     */
    var resetZoom = function() {
        currentZoom = 1;
        translateX = 0;
        translateY = 0;
        updateTransform();
        $('#fullscreen-modal .zoom-level').text('100%');
        $('#fullscreen-modal .modal-content').removeClass('zoomed');
    };

    /**
     * Start dragging
     * @param {Event} e - Mouse/Touch event
     */
    var startDrag = function(e) {
        if (currentZoom > 1) {
            isDragging = true;
            var pos = getEventPos(e);
            startX = pos.x - translateX;
            startY = pos.y - translateY;
            $('#fullscreen-modal .modal-content').addClass('zoomed');
        }
    };

    /**
     * Handle dragging
     * @param {Event} e - Mouse/Touch event
     */
    var drag = function(e) {
        if (isDragging) {
            e.preventDefault();
            var pos = getEventPos(e);
            translateX = pos.x - startX;
            translateY = pos.y - startY;

            // Calculate max translation based on image dimensions and zoom level
            var modalContent = $('#fullscreen-modal .modal-content');
            var imgWidth = modalContent[0].naturalWidth;
            var imgHeight = modalContent[0].naturalHeight;
            var maxTranslateX = (imgWidth * currentZoom - imgWidth) / 2;
            var maxTranslateY = (imgHeight * currentZoom - imgHeight) / 2;

            // Apply limits based on image dimensions
            translateX = Math.min(Math.max(translateX, -maxTranslateX), maxTranslateX);
            translateY = Math.min(Math.max(translateY, -maxTranslateY), maxTranslateY);

            updateTransform();
        }
    };

    /**
     * Stop dragging
     */
    var stopDrag = function() {
        isDragging = false;
        if (currentZoom === 1) {
            $('#fullscreen-modal .modal-content').removeClass('zoomed');
        }
    };

    /**
     * Get event position for both mouse and touch events
     * @param {Event} e - Mouse/Touch event
     * @return {Object} Position with x and y coordinates
     */
    var getEventPos = function(e) {
        if (e.type.startsWith('touch')) {
            var touch = e.originalEvent.touches[0] || e.originalEvent.changedTouches[0];
            return {
                x: touch.clientX,
                y: touch.clientY
            };
        }
        return {
            x: e.clientX,
            y: e.clientY
        };
    };

    /**
     * Update transform for zoom and pan
     */
    var updateTransform = function() {
        $('#fullscreen-modal .modal-content').css('transform',
            'translate(-50%, -50%) translate(' + translateX + 'px, ' + translateY + 'px) scale(' + currentZoom + ')');
    };

    return {
        init: init
    };
});
