<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_imagefullscreen;

/**
 * Text filter that adds fullscreen functionality to images.
 *
 * This filter detects img tags in content and adds a fullscreen icon overlay
 * that allows users to view images in fullscreen mode. The filter can be
 * configured to work with specific activity modules.
 *
 * @package    filter_imagefullscreen
 * @copyright  2025 Sandip R <radadiyasandip89@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class text_filter extends \core_filters\text_filter {

    #[\Override]
    public function filter($text, array $options = []) {
        global $PAGE;

        // Check if the text is a string and not empty.
        if (!is_string($text) || empty($text)) {
            return $text;
        }

        if (stripos($text, '</img>') === false && stripos($text, '<img') === false) {
            return $text;
        }

        // Only apply filter if we are in an activity (module) context.
        if ($this->context->contextlevel != CONTEXT_MODULE) {
            return $text;
        }

        // Check if we're in an activity and if it's enabled.
        // Get the course module info from context.
        $cm = get_coursemodule_from_id(null, $this->context->instanceid, 0, false, IGNORE_MISSING);
        if (!$cm) {
            return $text;
        }

        // Get enabled activities from config.
        $enabledactivities = get_config('filter_imagefullscreen', 'enabledactivities');

        // If no settings, enable only for book.
        if (empty($enabledactivities)) {
            $isenabled = ($cm->modname === 'book');
        } else {
            // Convert config value to array if it's not already.
            if (!is_array($enabledactivities)) {
                $enabledactivities = explode(',', $enabledactivities);
            }

            // Check if current module is in enabled list.
            $isenabled = in_array('mod_' . $cm->modname, $enabledactivities);
        }

        // If current module is not enabled, return unmodified text.
        if (!$isenabled) {
            return $text;
        }

        $this->setup($PAGE, $options['context'] ?? null);

        // First handle images within spans.
        $newtext = preg_replace_callback(
            '/<span[^>]*>(<img\s[^>]*src="[^"]*"[^>]*>)<\/span>/i',
            function ($matches) {
                $imagetag = $matches[1];
                $alttitle = $this->extract_alt_attribute($imagetag);
                $fullscreenicon = $this->render_fullscreen_icon();
                return $this->render_span_image($imagetag, $fullscreenicon, $alttitle);
            },
            $text
        );

        // Then handle standalone images - skip those already in containers.
        $pattern = '/<img\s[^>]*src="[^"]*"[^>]*>/i';
        if (preg_match_all($pattern, $newtext, $matches, PREG_OFFSET_CAPTURE)) {
            $offset = 0;
            foreach ($matches[0] as $match) {
                $img = $match[0];
                $pos = $match[1] + $offset;

                // Check if this image is already wrapped in our container.
                $before = substr($newtext, 0, $pos);
                $after = substr($newtext, $pos + strlen($img));

                // Skip if already in a span or already in our container.
                if (preg_match('/<span[^>]*>$/i', $before) && preg_match('/^<\/span>/i', $after)) {
                    continue;
                }

                if (preg_match('/<div[^>]*class="[^"]*image-fullscreen-container[^"]*"[^>]*>$/i', $before)) {
                    continue;
                }

                $alttitle = $this->extract_alt_attribute($img);
                $fullscreenicon = $this->render_fullscreen_icon();
                $replacement = $this->render_image_container($img, $fullscreenicon, $alttitle);

                $newtext = substr_replace($newtext, $replacement, $pos, strlen($img));
                $offset += strlen($replacement) - strlen($img);
            }
        }

        return $newtext;
    }

    #[\Override]
    public function setup($page, $context) {
        static $jsinitialised = false;

        if (!$jsinitialised) {
            $page->requires->js_call_amd('filter_imagefullscreen/fullscreen', 'init');
            $jsinitialised = true;
        }

        // Make sure parent is properly setup.
        parent::setup($page, $context);
    }

    /**
     * Render the fullscreen icon using Mustache template.
     *
     * @return string HTML for the fullscreen icon
     */
    private function render_fullscreen_icon() {
        global $OUTPUT;
        return $OUTPUT->render_from_template('filter_imagefullscreen/fullscreen_icon', []);
    }

    /**
     * Extract alt attribute from image tag for use as data-title.
     *
     * @param string $imagetag The image HTML tag
     * @return string The alt attribute value or empty string
     */
    private function extract_alt_attribute($imagetag) {
        if (preg_match('/alt\s*=\s*["\']([^"\']*)["\']/', $imagetag, $matches)) {
            return htmlspecialchars($matches[1], ENT_QUOTES, 'UTF-8');
        }
        return '';
    }

    /**
     * Render image container with fullscreen icon overlay using Mustache template.
     *
     * @param string $imagetag The original image HTML tag
     * @param string $fullscreenicon The fullscreen icon HTML
     * @param string $alttitle The alt text for data-title attribute
     * @return string Rendered HTML
     */
    private function render_image_container($imagetag, $fullscreenicon, $alttitle = '') {
        global $OUTPUT;
        return $OUTPUT->render_from_template('filter_imagefullscreen/image_container', [
            'imagetag' => $imagetag,
            'fullscreenicon' => $fullscreenicon,
            'alttitle' => $alttitle,
        ]);
    }

    /**
     * Render image within span with fullscreen icon using Mustache template.
     *
     * @param string $imagetag The original image HTML tag
     * @param string $fullscreenicon The fullscreen icon HTML
     * @param string $alttitle The alt text for data-title attribute
     * @return string Rendered HTML
     */
    private function render_span_image($imagetag, $fullscreenicon, $alttitle = '') {
        global $OUTPUT;
        return $OUTPUT->render_from_template('filter_imagefullscreen/span_image', [
            'imagetag' => $imagetag,
            'fullscreenicon' => $fullscreenicon,
            'alttitle' => $alttitle,
        ]);
    }
}
