<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_imagefullscreen;

/**
 * Tests for the filter_imagefullscreen text filter.
 *
 * @package    filter_imagefullscreen
 * @category   test
 * @copyright  2025 Sandip R <radadiyasandip89@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @covers \filter_imagefullscreen\text_filter
 */
final class text_filter_test extends \advanced_testcase {

    /**
     * Test that the filter applies to activity contexts only.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_only_applies_to_activity_contexts(): void {
        // Test in system context - filter should not apply.
        $systemcontext = \core\context\system::instance();
        $filter = new text_filter($systemcontext, []);
        $input = '<img src="test.jpg" alt="Test image">';
        $result = $filter->filter($input);
        $this->assertEquals($input, $result);

        // Test in course context - filter should not apply.
        $this->resetAfterTest(true);
        $course = $this->getDataGenerator()->create_course();
        $coursecontext = \core\context\course::instance($course->id);
        $filter = new text_filter($coursecontext, []);
        $result = $filter->filter($input);
        $this->assertEquals($input, $result);

        // Test in module context - filter should apply.
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $modulecontext = \core\context\module::instance($module->cmid);
        $filter = new text_filter($modulecontext, []);
        $result = $filter->filter($input);
        $this->assertNotEquals($input, $result);

        // Check that the result contains the fullscreen functionality.
        $this->assertMatchesRegularExpression('/image-fullscreen-container/', $result);
    }

    /**
     * Test the filter with basic image processing.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_basic_image(): void {
        $this->resetAfterTest(true);

        // Create a course and module for testing in activity context.
        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test basic image with alt text.
        $input = '<img src="test.jpg" alt="Test image">';
        $result = $filter->filter($input);

        $this->assertNotEquals($input, $result);
        $this->assertMatchesRegularExpression('/image-fullscreen-container/', $result);
        $this->assertMatchesRegularExpression('/fullscreen-icon/', $result);
        $this->assertMatchesRegularExpression('/data-title="Test image"/', $result);
    }

    /**
     * Test the filter with image without alt text.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_image_without_alt(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test image without alt text.
        $input = '<img src="test.jpg">';
        $result = $filter->filter($input);

        $this->assertNotEquals($input, $result);
        $this->assertMatchesRegularExpression('/image-fullscreen-container/', $result);
        $this->assertMatchesRegularExpression('/fullscreen-icon/', $result);
        // Should not have data-title when no alt text.
        $this->assertDoesNotMatchRegularExpression('/data-title/', $result);
    }

    /**
     * Test the filter preserves image attributes.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_preserves_attributes(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test image with multiple attributes.
        $input = '<img src="test.jpg" alt="Test" class="custom-class" id="myimg" width="100">';
        $result = $filter->filter($input);

        $this->assertNotEquals($input, $result);
        $this->assertMatchesRegularExpression('/class="custom-class"/', $result);
        $this->assertMatchesRegularExpression('/id="myimg"/', $result);
        $this->assertMatchesRegularExpression('/width="100"/', $result);
        $this->assertMatchesRegularExpression('/data-title="Test"/', $result);
    }

    /**
     * Test that already wrapped images are not modified.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_already_wrapped(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test image already in fullscreen container.
        $input = '<div class="image-fullscreen-container"><img src="test.jpg" alt="Test"></div>';
        $result = $filter->filter($input);

        $this->assertEquals($input, $result);
    }

    /**
     * Test the filter with multiple images.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_multiple_images(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test multiple images.
        $input = '<p><img src="test1.jpg" alt="Image 1"><img src="test2.jpg" alt="Image 2"></p>';
        $result = $filter->filter($input);

        $this->assertNotEquals($input, $result);
        // Should contain two fullscreen containers.
        $this->assertEquals(2, substr_count($result, 'image-fullscreen-container'));
        $this->assertEquals(2, substr_count($result, 'fullscreen-icon'));
        $this->assertMatchesRegularExpression('/data-title="Image 1"/', $result);
        $this->assertMatchesRegularExpression('/data-title="Image 2"/', $result);
    }

    /**
     * Test the filter with text content without images.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_no_images(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Test text with no images.
        $input = '<p>This is just text with no images.</p>';
        $result = $filter->filter($input);

        $this->assertEquals($input, $result);
    }

    /**
     * Test the filter preserves HTML structure.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_preserves_structure(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        $input = '<div class="wrapper"><p>Before</p><img src="test.jpg" alt="Test"><p>After</p></div>';
        $result = $filter->filter($input);

        // Check that structure is preserved.
        $this->assertMatchesRegularExpression('/<div class="wrapper">/', $result);
        $this->assertMatchesRegularExpression('/<p>Before<\/p>/', $result);
        $this->assertMatchesRegularExpression('/<p>After<\/p>/', $result);
        $this->assertMatchesRegularExpression('/image-fullscreen-container/', $result);
    }

    /**
     * Test filter performance with multiple images.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_performance(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();
        $module = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $context = \core\context\module::instance($module->cmid);

        $filter = new text_filter($context, []);

        // Create content with many images.
        $images = '';
        for ($i = 1; $i <= 20; $i++) {
            $images .= "<img src=\"image{$i}.jpg\" alt=\"Image {$i}\">";
        }
        $input = '<div>' . $images . '</div>';

        $starttime = microtime(true);
        $result = $filter->filter($input);
        $endtime = microtime(true);

        // Should complete quickly.
        $this->assertLessThan(1.0, $endtime - $starttime);
        // Check all images were processed.
        $this->assertEquals(20, substr_count($result, 'image-fullscreen-container'));
    }

    /**
     * Test that the filter respects the enabled activities setting.
     *
     * @covers \filter_imagefullscreen\text_filter::filter
     */
    public function test_filter_respects_enabled_activities_setting(): void {
        $this->resetAfterTest(true);

        $course = $this->getDataGenerator()->create_course();

        // Test with book module (should be enabled by default).
        $bookmodule = $this->getDataGenerator()->create_module('book', ['course' => $course->id]);
        $bookcontext = \core\context\module::instance($bookmodule->cmid);
        $filter = new text_filter($bookcontext, []);
        $input = '<img src="test.jpg" alt="Test image">';
        $result = $filter->filter($input);
        $this->assertNotEquals($input, $result, 'Filter should apply to book module by default');

        // Test with page module (should not be enabled by default).
        $pagemodule = $this->getDataGenerator()->create_module('page', ['course' => $course->id]);
        $pagecontext = \core\context\module::instance($pagemodule->cmid);
        $filter = new text_filter($pagecontext, []);
        $result = $filter->filter($input);
        $this->assertEquals($input, $result, 'Filter should not apply to page module by default');

        // Enable page module in settings.
        set_config('enabledactivities', 'mod_book,mod_page', 'filter_imagefullscreen');

        // Now test with page module again (should be enabled).
        $filter = new text_filter($pagecontext, []);
        $result = $filter->filter($input);
        $this->assertNotEquals($input, $result, 'Filter should apply to page module when enabled in settings');
    }
}
