<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

namespace local_savian_ai\content;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/question/engine/bank.php');
require_once($CFG->libdir . '/questionlib.php');

/**
 * Question Bank Creator - Imports AI-generated questions into Moodle
 *
 * Supports multichoice, true/false, short answer, essay, and matching question types.
 * Handles question formatting, answer processing, and feedback field population.
 *
 * @package    local_savian_ai
 * @copyright  2026 Savian AI
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class qbank_creator {

    /**
     * Add generated questions to Moodle Question Bank
     *
     * @param array $questions Array of question data from Savian AI
     * @param int $courseid Course ID
     * @param int $categoryid Question category ID (0 = create new)
     * @return array Results with success/failed lists
     */
    public function add_to_question_bank($questions, $courseid, $categoryid = 0) {
        global $DB, $CFG, $USER;

        $results = ['success' => [], 'failed' => []];

        // Get or create question category
        $context = \context_course::instance($courseid);
        if ($categoryid === 0) {
            $categoryid = $this->get_or_create_savian_category($context->id, $courseid);
        }

        foreach ($questions as $question_data) {
            try {
                $question_id = $this->create_question($question_data, $categoryid, $context);
                $results['success'][] = [
                    'name' => $question_data->name ?? $question_data->questiontext ?? 'Untitled',
                    'question_id' => $question_id,
                    'type' => $question_data->type ?? 'unknown',
                ];
            } catch (\Exception $e) {
                $results['failed'][] = [
                    'name' => $question_data->name ?? $question_data->questiontext ?? 'Untitled',
                    'error' => $e->getMessage(),
                ];
            }
        }

        return $results;
    }

    /**
     * Get or create Savian AI question category
     *
     * @param int $contextid Context ID
     * @param int $courseid Course ID
     * @return int Category ID
     */
    protected function get_or_create_savian_category($contextid, $courseid) {
        global $DB;

        // Try to find existing category
        $category = $DB->get_record('question_categories', [
            'contextid' => $contextid,
            'name' => 'Savian AI Generated',
        ]);

        if ($category) {
            return $category->id;
        }

        // Create new category
        $newcategory = new \stdClass();
        $newcategory->name = 'Savian AI Generated';
        $newcategory->contextid = $contextid;
        $newcategory->info = 'Questions generated by Savian AI';
        $newcategory->infoformat = FORMAT_HTML;
        $newcategory->stamp = make_unique_id_code();
        $newcategory->parent = 0;
        $newcategory->sortorder = 999;

        return $DB->insert_record('question_categories', $newcategory);
    }

    /**
     * Create a single question from Savian format
     *
     * @param object $data Question data
     * @param int $categoryid Category ID
     * @param object $context Context
     * @return int Question ID
     */
    protected function create_question($data, $categoryid, $context) {
        global $USER;

        // Map Savian types to Moodle qtypes
        $type_map = [
            'multichoice' => 'multichoice',
            'truefalse' => 'truefalse',
            'shortanswer' => 'shortanswer',
            'essay' => 'essay',
            'matching' => 'match',
        ];

        $qtype = $type_map[$data->type] ?? 'multichoice';

        // Build question form data
        $fromform = new \stdClass();
        $fromform->category = $categoryid . ',' . $context->id;
        $fromform->name = $this->truncate_name($data->name ?? $data->questiontext);
        $fromform->questiontext = ['text' => $this->extract_string($data->questiontext ?? ''), 'format' => FORMAT_HTML];
        $fromform->generalfeedback = ['text' => $this->extract_string($data->generalfeedback ?? ''), 'format' => FORMAT_HTML];
        $fromform->defaultmark = 1;
        $fromform->penalty = 0.3333333;

        // Add type-specific fields
        switch ($qtype) {
            case 'multichoice':
                $fromform = $this->build_multichoice($fromform, $data);
                break;
            case 'truefalse':
                $fromform = $this->build_truefalse($fromform, $data);
                break;
            case 'shortanswer':
                $fromform = $this->build_shortanswer($fromform, $data);
                break;
            case 'essay':
                $fromform = $this->build_essay($fromform, $data);
                break;
            case 'match':
                $fromform = $this->build_matching($fromform, $data);
                break;
        }

        // Save question using Moodle's question bank API
        $question = new \stdClass();
        $question->category = $categoryid;
        $question->qtype = $qtype;
        $question->createdby = $USER->id;
        $question->modifiedby = $USER->id;
        $question->timecreated = time();
        $question->timemodified = time();

        $question = \question_bank::get_qtype($qtype)->save_question($question, $fromform);

        return $question->id;
    }

    /**
     * Build multichoice question fields
     */
    protected function build_multichoice(\stdClass $fromform, $data) {
        $fromform->single = 1;  // Single answer
        $fromform->shuffleanswers = 1;
        $fromform->answernumbering = 'abc';
        $fromform->correctfeedback = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->partiallycorrectfeedback = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->incorrectfeedback = ['text' => '', 'format' => FORMAT_HTML];

        $fromform->answer = [];
        $fromform->fraction = [];
        $fromform->feedback = [];

        foreach ($data->answers as $idx => $answer) {
            $fromform->answer[$idx] = ['text' => $this->extract_string($answer->text ?? ''), 'format' => FORMAT_HTML];
            $fromform->fraction[$idx] = (float)(($answer->fraction ?? 0) / 100);  // Convert to 0-1
            $fromform->feedback[$idx] = ['text' => $this->extract_string($answer->feedback ?? ''), 'format' => FORMAT_HTML];
        }

        return $fromform;
    }

    /**
     * Build true/false question fields
     */
    protected function build_truefalse(\stdClass $fromform, $data) {
        // Find the correct answer
        $correct = 0;
        foreach ($data->answers as $answer) {
            if (($answer->fraction ?? 0) === 100) {
                $correct = (strtolower($this->extract_string($answer->text ?? '')) === 'true') ? 1 : 0;
                break;
            }
        }

        $fromform->correctanswer = $correct;
        $fromform->feedbacktrue = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->feedbackfalse = ['text' => '', 'format' => FORMAT_HTML];

        // Add required feedback fields
        $fromform->correctfeedback = ['text' => 'Correct!', 'format' => FORMAT_HTML];
        $fromform->partiallycorrectfeedback = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->incorrectfeedback = ['text' => 'Incorrect. Please review the material.', 'format' => FORMAT_HTML];

        return $fromform;
    }

    /**
     * Build short answer question fields
     */
    protected function build_shortanswer(\stdClass $fromform, $data) {
        $fromform->usecase = 0;  // Case insensitive

        $fromform->answer = [];
        $fromform->fraction = [];
        $fromform->feedback = [];

        foreach ($data->answers as $idx => $answer) {
            $fromform->answer[$idx] = $this->extract_string($answer->text ?? '');
            $fromform->fraction[$idx] = (float)(($answer->fraction ?? 100) / 100);
            $fromform->feedback[$idx] = ['text' => $this->extract_string($answer->feedback ?? ''), 'format' => FORMAT_HTML];
        }

        // Add required feedback fields
        $fromform->correctfeedback = ['text' => 'Correct!', 'format' => FORMAT_HTML];
        $fromform->partiallycorrectfeedback = ['text' => 'Partially correct.', 'format' => FORMAT_HTML];
        $fromform->incorrectfeedback = ['text' => 'Incorrect. Please review the material.', 'format' => FORMAT_HTML];

        return $fromform;
    }

    /**
     * Build essay question fields
     */
    protected function build_essay(\stdClass $fromform, $data) {
        $fromform->responseformat = 'editor';  // HTML editor
        $fromform->responserequired = 1;
        $fromform->responsefieldlines = 15;
        $fromform->attachments = 0;
        $fromform->attachmentsrequired = 0;
        $fromform->graderinfo = ['text' => $this->extract_string($data->graderinfo ?? ''), 'format' => FORMAT_HTML];
        $fromform->responsetemplate = ['text' => '', 'format' => FORMAT_HTML];

        // Add feedback fields (may not be used but prevents warnings)
        $fromform->correctfeedback = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->partiallycorrectfeedback = ['text' => '', 'format' => FORMAT_HTML];
        $fromform->incorrectfeedback = ['text' => '', 'format' => FORMAT_HTML];

        return $fromform;
    }

    /**
     * Build matching question fields
     */
    protected function build_matching(\stdClass $fromform, $data) {
        $fromform->shuffleanswers = 1;

        $fromform->subquestions = [];
        $fromform->subanswers = [];

        if (isset($data->subquestions)) {
            foreach ($data->subquestions as $idx => $subq) {
                $fromform->subquestions[$idx] = ['text' => $this->extract_string($subq->text ?? ''), 'format' => FORMAT_HTML];
                $fromform->subanswers[$idx] = $this->extract_string($subq->answer ?? '');
            }
        }

        // Add feedback fields (prevents warnings)
        $fromform->correctfeedback = ['text' => 'Well done!', 'format' => FORMAT_HTML];
        $fromform->partiallycorrectfeedback = ['text' => 'Partially correct.', 'format' => FORMAT_HTML];
        $fromform->incorrectfeedback = ['text' => 'Please review the material.', 'format' => FORMAT_HTML];

        return $fromform;
    }

    /**
     * Safely extract string from data (handles nested objects)
     */
    protected function extract_string($value, $default = '') {
        if (is_string($value)) {
            return $value;
        }
        if (is_object($value) || is_array($value)) {
            // Try to find 'en' key for bilingual content
            if (is_object($value) && isset($value->en)) {
                return (string)$value->en;
            }
            if (is_array($value) && isset($value['en'])) {
                return (string)$value['en'];
            }
            // Fallback to JSON
            return json_encode($value);
        }
        return (string)$default;
    }

    /**
     * Create a readable question name (not full text)
     */
    protected function truncate_name($name) {
        $name = $this->extract_string($name);
        $name = strip_tags($name);
        $name = trim($name);

        // If too long, create a meaningful summary
        if (strlen($name) > 80) {
            // Take first 10 words
            $words = explode(' ', $name);
            $short_words = array_slice($words, 0, 10);
            $name = implode(' ', $short_words);

            // Clean up any trailing punctuation
            $name = rtrim($name, '.,;:?!');
            $name .= '...';
        }

        // Ensure within Moodle's 255 char limit
        $name = substr($name, 0, 255);
        return $name;
    }
}
