<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * HelloAsso payment gateway logger class.
 *
 * @package    paygw_helloasso
 * @copyright  2025 Sebastien Chassande-Barrioz <chassande@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace paygw_helloasso;

/**
 * Logger class for HelloAsso payment gateway.
 *
 * @package    paygw_helloasso
 * @copyright  2025 Sebastien Chassande-Barrioz <chassande@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class logger {

    /**
     * Log une action du plugin
     *
     * @param int $paymentid ID du paiement
     * @param int $userid ID de l'utilisateur
     * @param string $action Type d'action
     * @param string $status Statut (success, error, fraud_detected)
     * @param float $amount Montant du paiement
     * @param string $message Message détaillé
     * @param int $responsecode Code HTTP de réponse
     * @param string $reference Référence HelloAsso
     */
    public static function log_action($paymentid, $userid, $action, $status, $amount,
        $message = '', $responsecode = null, $reference = '') {
        global $DB;

        $log = new \stdClass();
        $log->paymentid = $paymentid;
        $log->userid = $userid;
        $log->action = $action;
        $log->status = $status;
        $log->amount = $amount;
        $log->reference = $reference;
        $log->message = $message;
        $log->response_code = $responsecode;
        $log->ip_address = getremoteaddr();
        $log->timecreated = time();

        $DB->insert_record('paygw_helloasso_logs', $log);
    }

    /**
     * Récupère les logs d'un paiement
     *
     * @param int $paymentid ID du paiement
     * @return array Array of log records
     */
    public static function get_payment_logs($paymentid) {
        global $DB;
        return $DB->get_records('paygw_helloasso_logs', ['paymentid' => $paymentid], 'timecreated DESC');
    }

    /**
     * Récupère les logs avec erreurs
     *
     * @param int $limit Nombre maximum de logs à retourner
     * @return array Array of error log records
     */
    public static function get_error_logs($limit = 100) {
        global $DB;
        return $DB->get_records('paygw_helloasso_logs',
            ['status' => 'error'],
            'timecreated DESC',
            '*',
            0,
            $limit
        );
    }

    /**
     * Récupère les paiements suspects (fraude)
     *
     * @param int $limit Nombre maximum d'alertes à retourner
     * @return array Array of fraud alert log records
     */
    public static function get_fraud_alerts($limit = 50) {
        global $DB;
        return $DB->get_records('paygw_helloasso_logs',
            ['status' => 'fraud_detected'],
            'timecreated DESC',
            '*',
            0,
            $limit
        );
    }
}
