<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace report_sphorphanedfiles\View;

use stdClass;
use moodle_database;
use context_course;
use report_sphorphanedfiles\Files\Files;
use report_sphorphanedfiles\Files\FileInfo;
use report_sphorphanedfiles\Manager;
use report_sphorphanedfiles\Misc;
use report_sphorphanedfiles\HTML;
use UnexpectedValueException;

/**
 * Class OrphanedView
 *
 * @package report_sphorphanedfiles
 * @copyright   Schulportal Hessen (SPH)
 * @author      Andreas Schenkel <andreas.schenkel@schulportal.hessen.de>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class OrphanedView {
    private $page;

    /**
     * @var int
     */
    private $courseId;

    /**
     * @var stdClass
     */
    private $user;

    /**
     * @var Manager
     */
    private $apiM;

    /**
     * @var bool
     */
    private $afterDeletion = false;

    /**
     * Indicates if a course uses gridformat-plugin
     *
     * @var bool
     */
    private $courseFormatGridEnabled = false;

    /**
     * OrphanedView constructor.
     * @param moodle_database $db
     * @param int $courseId
     * @param moodle_page $page
     * @param bootstrap_renderer $output
     * @param stdClass $user
     */
    public function __construct($db, int $courseId, $page, $output, $user) {
        $this->courseId = $courseId;
        $this->user = $user;
        $this->apiM = new Manager($db);

        $course = $this->apiM->database()->dataFiles()->getCourse($courseId);

        $this->page = new Page($page, $course, $courseId, $output);
    }

    public function getPage(): Page {
        return $this->page;
    }

    /**
     * If the page is opened with a POST request,
     * this means the user has confirmed to delete a single orphaned file
     * We have to be sure that the submitted filedata were not manipulated, the file belongs to the course and the user
     * is allowed to delete the file.
     *
     * @return void
     * @throws coding_exception
     * @throws moodle_exception
     * @throws require_login_exception
     */
    public function deleteOrphanedFile(): void {
        // Read all post-parameter as required parameter and for each parameter check-
        // - type
        // - length
        // ToDo: some more securitychecks on the Post-Parameter.
        $pathnamehash = required_param('pathnamehash', PARAM_ALPHANUM);
        if (strlen($pathnamehash) > 40) {
            throw new UnexpectedValueException('wrong pathnamehash');
            return;
        }

        $contextId = required_param('contextId', PARAM_INT);

        $component = required_param('component', PARAM_TEXT);
        if (strlen($component) > 100) {
            throw new UnexpectedValueException('wrong component');
            return;
        }

        $filearea = required_param('filearea', PARAM_TEXT);
        if (strlen($filearea) > 50) {
            throw new UnexpectedValueException('wrong filearea');
            return;
        }

        $itemId = required_param('itemId', PARAM_INT);

        $filepath = required_param('filepath', PARAM_TEXT);
        if (strlen($filepath) > 255) {
            throw new UnexpectedValueException('wrong filepath');
            return;
        }

        $filename = required_param('filename', PARAM_TEXT);
        if (strlen($filename) > 255) {
            throw new UnexpectedValueException('wrong filename');
            return;
        }

        $postDataFile = [];
        $postDataFile['pathnamehash'] = $pathnamehash;
        $postDataFile['contextId'] = $contextId;
        $postDataFile['component'] = $component;
        $postDataFile['filearea'] = $filearea;
        $postDataFile['itemId'] = $itemId;
        $postDataFile['filepath'] = $filepath;
        $postDataFile['filename'] = $filename;
        $serialisation_PostDataFile = (new FileInfo($postDataFile))->toString();

        // Get the file that might should be deleted.
        $fileToBeDeleted = (new Files())->getFileStorage()->get_file_by_hash($pathnamehash);
        if (!$fileToBeDeleted) {
            // If file was already deleted.
            throw new UnexpectedValueException('file not found');
            return;
        }

        // Get the contextid of the file.
        $dataFileToBeDeleted = [];
        $dataFileToBeDeleted['pathnamehash'] = $fileToBeDeleted->get_pathnamehash();
        $dataFileToBeDeleted['contextId'] = $fileToBeDeleted->get_contextid();
        $dataFileToBeDeleted['component'] = $fileToBeDeleted->get_component();
        $dataFileToBeDeleted['filearea'] = $fileToBeDeleted->get_filearea();
        $dataFileToBeDeleted['itemId'] = $fileToBeDeleted->get_itemid();
        $dataFileToBeDeleted['filepath'] = $fileToBeDeleted->get_filepath();
        $dataFileToBeDeleted['filename'] = $fileToBeDeleted->get_filename();
        // Serialize in order to be able to compare with the $fileID.
        $serialisation_FileToBeDeleted = (new FileInfo($dataFileToBeDeleted))->toString();

        // Compare file from Post with $fileToBeDeleted-Information.
        if ($serialisation_FileToBeDeleted != $serialisation_PostDataFile) {
            // Files are not equal ...
            throw new UnexpectedValueException('wrong value found');
            return;
        }

        $this->afterDeletion = $this->apiM->files()->deleteFileInCourse(
            $this->apiM->security(),
            $fileToBeDeleted,
            $this->user,
            $this->courseId
        );
    }

    public function listOrphansForSection($sectionInfo) {
        $courseContextId = context_course::instance($this->courseId)->id;

        $viewOrphanedFiles = [];
        $viewOrphanedFiles = $this->apiM->handler()->sectionSummaryHandler()->getViewOrphanedFiles(
            $viewOrphanedFiles,
            $courseContextId,
            $sectionInfo,
            $this->user,
            $this->courseId,
            '' // Intentionally left blank: In case of a section summary, there is no iconHtml information.
        );

        $modInfo = $sectionInfo->modinfo;

        // ToDo: Redundante Itteration refactorn.
        foreach ($modInfo->get_instances() as $moduleinstances) {
            foreach ($moduleinstances as $cm) {
                if ($sectionInfo->id === $cm->sectionid) {
                    if ($cm->deletioninprogress !== '1') {
                        if ($this->apiM->handler()->hasHandlerFor($cm)) {
                            $viewOrphanedFiles = $this->apiM->handler()->getHandlerFor($cm)
                                ->bind($this->user, $this->courseId, $cm, $this->getPage())
                                ->addOrphans($viewOrphanedFiles);
                        }
                    }
                }
            }
        }

        return $viewOrphanedFiles;
    }

    public function createOrphansList($sectionInfo): string {
        $viewOrphanedFiles = $this->listOrphansForSection($sectionInfo);
        $cleanedViewOrphanedFiles = [];
        // Do not mark plugin gridlayout files as orphaned.
        foreach ($viewOrphanedFiles ?? [] as $viewOrphanedFile) {
            if (
                !($this->courseFormatGridEnabled && isset($viewOrphanedFile['isGridlayoutFile']) &&
                    $viewOrphanedFile['isGridlayoutFile'])
            ) {
                $cleanedViewOrphanedFiles[] = $viewOrphanedFile;
            }
        }

        if (!empty($cleanedViewOrphanedFiles)) {
            $translations = Misc::translate(
                ['isallowedtodeleteallfiles', 'description', 'isgridlayoutfilehint'],
                'report_sphorphanedfiles'
            );
            $translations['header'] = Misc::translate(
                ['modName', 'content', 'filename', 'preview', 'tool', 'moduleContent', 'code'],
                'report_sphorphanedfiles',
                'header.'
            );
            $data = [
                'orphanedFilesList' => $cleanedViewOrphanedFiles,
                'translation' => $translations,
            ];
            // Temporarily disabled: $dummy = json_encode($data).
            return $this->getPage()->getOutput()->render_from_template(
                'report_sphorphanedfiles/sectionTable',
                $data
            );
        }

        return "";
    }

    /**
     * Initialize.
     *
     * @throws coding_exception
     * @throws dml_exception
     * @throws moodle_exception
     * @throws require_login_exception
     */
    public function init() {
        if (isset($this->getPage()->getCourse()->format) && $this->getPage()->getCourse()->format === 'grid') {
            $this->courseFormatGridEnabled = true;
        }

        $userAllowedToDeleteFiles = $this->apiM->security()->isUserAllowedToDeleteFiles(
            $this->courseId,
            $this->user
        );
        echo $this->getPage()->getOutput()->header();
        // Render content above the table.
        $data = [
            'title' => $this->getPage()->getTitle(),
            '$userAllowedToDeleteFiles' => $userAllowedToDeleteFiles,
            'afterDeletion' => $this->afterDeletion,
            'deleteMessage' => get_string('deleteMessage', 'report_sphorphanedfiles'),
            'translation' => Misc::translate(
                ['isallowedtodeleteallfiles', 'description', 'isgridlayoutfilehint'],
                'report_sphorphanedfiles'
            ),
        ];
        // Temporarily disabled: $dummy = json_encode($data).
        echo $this->getPage()->getOutput()->render_from_template(
            'report_sphorphanedfiles/report',
            $data
        );

        // Now render each section.
        $sectionCounter = 0;
        foreach ($this->getPage()->getCourseInfo()->get_section_info_all() as $sectionInfo) {
            echo HTML::createSectionOverview(
                3,
                HTML::createSectionHeading($sectionInfo, $this->getPage()->getCourse(), $sectionCounter++),
                $this->createOrphansList($sectionInfo)
            );
        }

        echo $this->getPage()->getOutput()->footer();
    }
}
