<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * privacy provider for plagiarism_strike
 *
 * File         provider.php
 * Encoding     UTF-8
 *
 * @package     plagiarism_strike
 *
 * @copyright   Sebsoft.nl / Strikeplagiarism.com
 * @author      R.J. van Dongen <rogier@sebsoft.nl>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace plagiarism_strike\privacy;

defined('MOODLE_INTERNAL') || die();

use core_privacy\local\metadata\collection;
use core_privacy\local\request\contextlist;
use core_privacy\local\request\helper;
use core_privacy\local\request\writer;

require_once($CFG->dirroot . '/plagiarism/strike/classes/strikeremovedocument.php');

/**
 * privacy provider for plagiarism_strike
 *
 * @package     plagiarism_strike
 *
 * @copyright   Sebsoft.nl / Strikeplagiarism.com
 * @author      R.J. van Dongen <rogier@sebsoft.nl>
 * @author      Kamil Łuczak <kluczak@intersiec.com.pl>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class provider implements
// This plugin does store personal user data.
\core_privacy\local\metadata\provider, \core_plagiarism\privacy\plagiarism_provider {

    // This trait must be included to provide the relevant polyfill for the metadata provider.
    use \core_privacy\local\legacy_polyfill;

    // This trait must be included to provide the relevant polyfill for the plagirism provider.
    use \core_plagiarism\privacy\legacy_polyfill;

    /**
     * Return the fields which contain personal data.
     *
     * @param collection $collection a reference to the collection to use to store the metadata.
     * @return collection $collection the updated collection of metadata items.
     */
    public static function get_metadata(collection $collection): collection {

        $collection->link_subsystem(
                'core_files', 'privacy:metadata:core_files'
        );

        $collection->add_database_table('plagiarism_strike_files', [
            'userid' => 'privacy:metadata:userid',
            'relateduserid' => 'privacy:metadata:relateduserid',
            'title' => 'privacy:metadata:title',
            'author' => 'privacy:metadata:author',
            'coordinator' => 'privacy:metadata:coordinator',
            'documentkind' => 'privacy:metadata:documentkind',
            'filename' => 'privacy:metadata:filename',
            'guid' => 'privacy:metadata:guid',
            'md5sum' => 'privacy:metadata:md5sum',
            'reportready' => 'privacy:metadata:reportready',
            'indexed' => 'privacy:metadata:indexed',
            'factor1' => 'privacy:metadata:factor1',
            'factor2' => 'privacy:metadata:factor2',
            'factor3' => 'privacy:metadata:factor3',
            'factor4' => 'privacy:metadata:factor4',
            'factor5' => 'privacy:metadata:factor5',
            'timesubmitted' => 'privacy:metadata:ftimesubmitted',
            'timecreated' => 'privacy:metadata:ftimecreated',
            'timemodified' => 'privacy:metadata:ftimemodified',
                ], 'privacy:metadata:plagiarism_strike_files');

        $collection->add_database_table('plagiarism_strike_reports', [
            'strikefileid' => 'privacy:metadata:strikefileid',
            'report' => 'privacy:metadata:report',
            'guid' => 'privacy:metadata:guid',
            'timecreated' => 'privacy:metadata:timecreated',
            'timemodified' => 'privacy:metadata:timemodified',
                ], 'privacy:metadata:plagiarism_strike_reports');

        $collection->add_external_location_link('plagiarism_strike', [
            'data' => 'privacy:metadata:plagiarism_strike:data'
                ], 'privacy:metadata:plagiarism_strike:externalpurpose');

        return $collection;
    }

    /**
     * Get the list of contexts that contain user information for the specified user.
     *
     * @param int $userid the userid.
     * @return contextlist the list of contexts containing user info for the user.
     */
    public static function get_contexts_for_userid($userid) : contextlist {

        // Assign.
        $params = ['modulename' => 'assign',
            'contextlevel' => CONTEXT_MODULE,
            'userid' => $userid];

        $sql = "SELECT ctx.id
                  FROM {course_modules} cm
                  JOIN {modules} m ON cm.module = m.id AND m.name = :modulename
                  JOIN {assign} a ON cm.instance = a.id
                  JOIN {context} ctx ON cm.id = ctx.instanceid AND ctx.contextlevel = :contextlevel
             LEFT JOIN {plagiarism_strike_files} ps ON cm.instance = cm
                 WHERE ps.userid = :userid";

        $contextlist = new contextlist();
        $contextlist->add_from_sql($sql, $params);

        // Forum.
        $params = [
            'modname' => 'forum',
            'contextlevel' => CONTEXT_MODULE,
            'discussionuserid' => $userid,
        ];
        $sql = "SELECT c.id
                 FROM {context} c
           INNER JOIN {course_modules} cm ON cm.id = c.instanceid AND c.contextlevel = :contextlevel
           INNER JOIN {modules} m ON m.id = cm.module AND m.name = :modname
           INNER JOIN {forum} f ON f.id = cm.instance
            LEFT JOIN {plagiarism_strike_files} tf ON cm.instance = cm
            LEFT JOIN {forum_discussions} d ON d.forum = f.id
                WHERE (
                d.userid        = :discussionuserid
                )
        ";
        $contextlist->add_from_sql($sql, $params);

        // Workshop.
        $params = [
            'modname' => 'workshop',
            'contextlevel' => CONTEXT_MODULE,
            'submissionauthorid' => $userid,
        ];
        $sql = "SELECT c.id
                 FROM {context} c
           INNER JOIN {course_modules} cm ON cm.id = c.instanceid AND c.contextlevel = :contextlevel
           INNER JOIN {modules} m ON m.id = cm.module AND m.name = :modname
           INNER JOIN {workshop} w ON w.id = cm.instance
            LEFT JOIN {plagiarism_strike_files} tf ON cm.instance = cm
            LEFT JOIN {workshop_submissions} s ON s.workshopid = w.id
                WHERE (
                s.authorid        = :submissionauthorid
                )
        ";
        $contextlist->add_from_sql($sql, $params);

        return $contextlist;
    }

    /**
     * Export all plagiarism data from each plagiarism plugin for the specified userid and context.
     *
     * @param   int         $userid The user to export.
     * @param   \context    $context The context to export.
     * @param   array       $subcontext The subcontext within the context to export this information to.
     * @param   array       $linkarray The weird and wonderful link array used to display information for a specific item
     */
    public static function export_plagiarism_user_data($userid, \context $context, array $subcontext, array $linkarray) {
        global $DB;

        if (empty($userid)) {
            return;
        }

        $user = $DB->get_record('user', array('id' => $userid));

        $params = ['userid' => $user->id];

        $sql = "SELECT p.id,
                p.cm cm,
                p.relateduserid plagiarism_strike_relateduserid,
                p.title plagiarism_strike_title,
                p.author plagiarism_strike_author,
                p.coordinator plagiarism_strike_coordinator,
                p.documentkind plagiarism_strike_documentkind,
                p.filename plagiarism_strike_filename,
                p.guid plagiarism_strike_guid,
                p.md5sum plagiarism_strike_md5sum,
                p.reportready plagiarism_strike_reportready,
                p.indexed plagiarism_strike_indexed,
                p.factor1 plagiarism_strike_factor1,
                p.factor2 plagiarism_strike_factor2,
                p.factor3 plagiarism_strike_factor3,
                p.factor4 plagiarism_strike_factor4,
                p.factor5 plagiarism_strike_factor5,
                p.timesubmitted plagiarism_strike_submission_time,
                p.timecreated plagiarism_strike_submission_created,
                p.timemodified plagiarism_strike_modification_time,
                r.report plagiarism_strike_report,
                r.timecreated as plagiarism_strike_report_timecreated,
                r.timemodified as plagiarism_strike_report_timemodified
                  FROM {plagiarism_strike_files} p
                  LEFT JOIN {plagiarism_strike_reports} r ON r.strikefileid = p.id
                 WHERE userid = :userid";
        $submissions = $DB->get_records_sql($sql, $params);

        foreach ($submissions as $submission) {
            $context = \context_module::instance($submission->cm);
            self::_export_plagiarism_strike_data_for_user((array) $submission, $context, $user);
        }
    }

    /**
     * Export the supplied personal data for a single activity, along with any generic data or area files.
     *
     * @param array $submissiondata the personal data to export.
     * @param \context_module $context the module context.
     * @param \stdClass $user the user record
     */
    protected static function export_plagiarism_strike_data_for_user(array $submissiondata,
            \context_module $context, \stdClass $user) {
        // Fetch the generic module data.
        $contextdata = helper::get_context_data($context, $user);

        // Merge with module data and write it.
        $contextdata = (object) array_merge((array) $contextdata, $submissiondata);
        writer::with_context($context)->export_data([], $contextdata);

        // Write generic module intro files.
        helper::export_context_files($context, $user);
    }

    /**
     * Delete all data for all users in the specified context.
     *
     * @param \context $context the context to delete in.
     */
    public static function delete_plagiarism_for_context(\context $context) {
        global $DB;

        if (empty($context)) {
            return;
        }

        if (!$context instanceof \context_module) {
            return;
        }

        $strikeremovedocumentfromreferencedatabase = new \plagiarism_strike\strikeremovedocument();
        $strikeremovedocumentfromdatabase = new \plagiarism_strike\strikeremove();
        // Delete all reports and files from plagiarism.com.
        if ($records = $DB->get_records('plagiarism_strike_files', array('cm' => $context->instanceid))) {
            foreach ($records as $val) {
                $strikeremovedocumentfromreferencedatabase->remove_from_database($val->guid);
                $strikeremovedocumentfromdatabase->remove($val->guid);
                $DB->delete_records('plagiarism_strike_reports', ['guid' => $val->guid]);
            }
        }
        // Delete all submissions.
        $DB->delete_records('plagiarism_strike_files', ['cm' => $context->instanceid]);
    }

    /**
     * Delete all user information for the provided user and context.
     *
     * @param  int      $userid    The user to delete
     * @param  \context $context   The context to refine the deletion.
     */
    public static function delete_plagiarism_for_user($userid, \context $context) {
        global $DB;
        // Delete all reports and files from remote server.
        $strikeremovedocumentfromreferencedatabase = new \plagiarism_strike\strikeremovedocument();
        $strikeremovedocumentfromdatabase = new \plagiarism_strike\strikeremove();
        if ($records = $DB->get_records('plagiarism_strike_files', ['userid' => $userid])) {
            foreach ($records as $val) {
                $strikeremovedocumentfromreferencedatabase->remove_from_database($val->guid);
                $strikeremovedocumentfromdatabase->remove($val->guid);
                $DB->delete_records('plagiarism_strike_reports', ['guid' => $val->guid]);
            }
        }
        // Delete all submissions.
        $DB->delete_records('plagiarism_strike_files', ['userid' => $userid]);
    }

}
