<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * plagiarism_strike external API.
 *
 * @package    plagiarism_strike
 * @copyright  UAB "Vextur" <info@vextur.com>
 * @author     2024, Andrius Martinkėnas <andrius@vextur.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die;

require_once($CFG->libdir . '/externallib.php');

/**
 * plagiarism_strike external API.
 *
 * @package    plagiarism_strike
 * @copyright  UAB "Vextur" <info@vextur.com>
 * @author     2024, Andrius Martinkėnas <andrius@vextur.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class plagiarism_strike_external extends external_api
{

    /**
     * Describes the parameters for get_protocols.
     *
     * @return external_function_parameters
     * @since Moodle 3.3
     */
    public static function get_protocols_parameters()
    {
        return new external_function_parameters(
            array(
                'key' => new external_value(PARAM_TEXT, 'API key'),
                'server' => new external_value(PARAM_URL, 'Server URL')
            )
        );
    }

    /**
     * Save protocols to database.
     *
     * @return array of values
     * @since Moodle 3.3
     */
    public static function get_protocols($key, $server)
    {
        global $DB;
        self::validate_parameters(self::get_protocols_parameters(), array('key' => $key, 'server' => $server));

        $striketest = new \plagiarism_strike\strikegetprotocols();
        $connection = $striketest->get_protocols($key, $server);
        if (!$connection) {
            return [
                'response' => get_string('errorprotocol', 'plagiarism_strike'),
                'call' => false,
            ];
        } else {
            $currentids = $DB->get_records_menu('plagiarism_strike_protocols', null, '', 'protocoltypeid, id');
            $newids = [];
            foreach ($connection as $protocol) {
                $existingprotocol = $DB->get_record('plagiarism_strike_protocols', array('protocoltypeid' => $protocol['protocolTypeId']));
                $newids[] = $protocol['protocolTypeId'];
                if ($existingprotocol) {
                    // Update the record if it exists
                    $existingprotocol->protocolname = $protocol['protocolName'];
                    $existingprotocol->protocolfilename = $protocol['protocolFilename'];
                    $DB->update_record('plagiarism_strike_protocols', $existingprotocol);
                } else {
                    // Insert a new record if it does not exist
                    $newprotocol = new stdClass();
                    $newprotocol->protocoltypeid = $protocol['protocolTypeId'];
                    $newprotocol->protocolname = $protocol['protocolName'];
                    $newprotocol->protocolfilename = $protocol['protocolFilename'];
                    $DB->insert_record('plagiarism_strike_protocols', $newprotocol);
                }
            }

            $deleteids = array_diff(array_keys($currentids), $newids);

            if (!empty($deleteids)) {
                list($insql, $params) = $DB->get_in_or_equal($deleteids);
                $DB->delete_records_select('plagiarism_strike_protocols', "protocoltypeid $insql", $params);
            }
            $count = $DB->get_field_sql("SELECT COUNT(id) FROM {plagiarism_strike_protocols}");
            $response = get_string('countprotocols', 'plagiarism_strike') . $count;
        }

        return [
            'response' => $response,
            'call' => true,
        ];
    }

    /**
     * Describes the get_protocols return value.
     *
     * @return external_single_structure
     * @since Moodle 3.3
     */
    public static function get_protocols_returns()
    {
        return new external_single_structure(
            [
                'response' => new external_value(PARAM_TEXT, 'Response'),
                'call' => new external_value(PARAM_BOOL, 'Api call status'),
            ]
        );
    }
}

