<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Config class for plagiarism_strike
 *
 * File         config.php
 * Encoding     UTF-8
 *
 * @package     plagiarism_strike
 *
 * @copyright   Sebsoft.nl / Strikeplagiarism.com
 * @author      R.J. van Dongen <rogier@sebsoft.nl>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace plagiarism_strike;

defined('MOODLE_INTERNAL') || die();

/**
 * plagiarism_strike\strikefile
 *
 * @package     plagiarism_strike
 *
 * @copyright   Sebsoft.nl / Strikeplagiarism.com
 * @author      R.J. van Dongen <rogier@sebsoft.nl>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class strikefile {

    /**
     * DB Storage table
     * @var string
     */
    private static $table = 'plagiarism_strike_files';

    /**
     * table ID / record PKEY
     * @var int
     */
    public $id;
    /**
     * Course module ID
     * @var int
     */
    public $cm;
    /**
     * User ID
     * @var int
     */
    public $userid;
    /**
     * Related user ID
     * @var int
     */
    public $relateduserid;
    /**
     * Should this refer to a file on the moodle filesystem?
     * @var int
     */
    public $isstoredfile = 0;
    /**
     * Is analysis automated?
     * @var int
     */
    public $automatedanalysis = 0;
    /**
     * Should analysis be done for this record?
     * @var int
     */
    public $doanalysis = 0;
    /**
     * document title
     * @var string
     */
    public $title;
    /**
     * document author
     * @var string
     */
    public $author;
    /**
     * coordinator
     * @var string
     */
    public $coordinator;
    /**
     * documentkind
     * @var string
     */
    public $documentkind;
    /**
     * documentuserid
     * @var string
     */
    public $documentuserid;
    /**
     * assignmentid
     * @var string
     */
    public $assignmentid;
    /**
     * is report for submitted file ready?
     * @var int
     */
    public $reportready = 0;
    /**
     * is document indexed in STRIKE database?
     * @var int
     */
    public $indexed = 0;
    /**
     * Similarity Coefficient 1
     * @var int
     */
    public $factor1 = 0;
    /**
     * Similarity Coefficient 2
     * @var int
     */
    public $factor2 = 0;
    /**
     * Similarity Coefficient 3
     * @var int
     */
    public $factor3 = 0;
    /**
     * Similarity Coefficient 4
     * @var int
     */
    public $factor4 = 0;
    /**
     * Similarity Coefficient 5
     * @var int
     */
    public $factor5 = 0;
    /**
     * Character detection of foreign alphabet in document
     * @var int
     */
    public $foreignalphabetalert = 0;
    /**
     * MD5 checksum of document contents
     * @var string
     */
    public $md5sum;
    /**
     * document GUID
     * @var string
     */
    public $guid;
    /**
     * document content hash (SHA1, because that's Moodle's standard UGH)
     * @var string
     */
    public $contenthash;
    /**
     * document filename
     * @var string
     */
    public $filename;
    /**
     * document filepath
     * @var string
     */
    public $filepath;
    /**
     * document submission attempt
     * @var int
     */
    public $attempt = 0;
    /**
     * submission statuscode
     * @var string|int
     */
    public $statuscode;
    /**
     * submission error, if anything
     * @var string
     */
    public $errorresponse = null;
    /**
     * Time of document submission to strike
     * @var int
     */
    public $timesubmitted;
    /**
     * Time of record creation
     * @var int
     */
    public $timecreated;
    /**
     * Time of record modification
     * @var int
     */
    public $timemodified;
    /**
     * status file
     * @var int
     */
    public $deleted = 0;

    /**
     * Create a new instance given the parameters to fill the object with
     *
     * @param array|\stdClass $args key-values
     * @return \static
     */
    public static function create($args) {
        $self = new static();
        $self->load_data($args);
        return $self;
    }

    /**
     * Load the parameters to this object
     *
     * @param array|\stdClass $data key-values
     * @return \static
     */
    protected function load_data($data) {
        foreach ($data as $key => $value) {
            $m = "set_{$key}";
            if (method_exists($this, $m)) {
                call_user_func(array($this, $m), $value);
            } else if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Get the storage table
     * @return string Database storage table
     */
    public static function table() {
        return static::$table;
    }

    /**
     * Get object by id
     * @param int $id
     * @return static
     */
    public static function get_by_id($id) {
        global $DB;
        $rs = $DB->get_record(static::table(), array('id' => $id));
        return static::create($rs);
    }

    /**
     * Get object by guid
     * @param string $guid
     * @param bool $mustexist
     * @return static
     */
    public static function get_by_guid($guid, $mustexist = true) {
        global $DB;
        $conditions = array('guid' => $guid);
        $rs = $DB->get_record(static::table(), $conditions, '*', $mustexist ? MUST_EXIST : IGNORE_MISSING);
        if (empty($rs)) {
            $rs = $conditions;
        }
        return static::create($rs);
    }

    /**
     * Get one record based on given conditions
     * @param array $conditions
     * @return static
     */
    public static function get_one($conditions) {
        global $DB;
        $rs = $DB->get_record(static::table(), $conditions, '*', IGNORE_MULTIPLE);
        if (empty($rs)) {
            $rs = $conditions;
        }
        return static::create($rs);
    }

    /**
     * Get records based on given conditions
     * @param array|null $conditions
     * @return static[]
     */
    public static function get($conditions) {
        global $DB;
        $rs = $DB->get_records(static::table(), $conditions);
        $collection = [];
        foreach ($rs as $key => $record) {
            $collection[$key] = static::create($record);
        }
        return $collection;
    }

    /**
     * Get records for the given select statement
     *
     * @param string $select
     * @param array|null $params
     * @return static[]
     */
    public static function get_select($select, $params = null) {
        global $DB;
        $rs = $DB->get_records_select(static::table(), $select, $params);
        $collection = [];
        foreach ($rs as $key => $record) {
            $collection[$key] = static::create($record);
        }
        return $collection;
    }

    /**
     * Get records for the given query
     *
     * @param string $sql
     * @param array|null $params
     * @return static[]
     */
    public static function get_sql($sql, $params = null) {
        global $DB;
        $rs = $DB->get_records_sql($sql, $params);
        $collection = [];
        foreach ($rs as $key => $record) {
            $collection[$key] = static::create($record);
        }
        return $collection;
    }

    /**
     * Remove object from database
     * @return bool
     */
    public function delete() {
        global $DB;
        // Delete report if it exists.
        $DB->delete_records(strikereport::table(), array('strikefileid' => $this->id));
        return $DB->delete_records(static::table(), array('id' => $this->id));
    }

    /**
     * Set field deleted to true
     * @return bool
     */
    public function setStatusDeleted() {
        global $DB;
        // Delete report if it exists.
        $DB->delete_records(strikereport::table(), array('strikefileid' => $this->id));
        if($record = $DB->get_record(static::table(), array('id' => $this->id))) {
            $record->deleted = 1;
            $DB->update_record(static::table(), $record);
        }

        //$this->deleted = true;
        //return $this->update();
        return false;
    }

    /**
     * Save object to database
     * @return bool|int
     */
    public function save() {
        if (!empty($this->id)) {
            return $this->update();
        } else {
            return $this->insert();
        }
    }

    /**
     * Insert object in database
     * @return int primary key
     */
    protected function insert() {
        global $DB;
        $this->timecreated = time();
        $this->timemodified = $this->timecreated;
        $this->id = $DB->insert_record(static::table(), $this);
        return $this->id;
    }

    /**
     * Update object in database
     * @return bool
     */
    protected function update() {
        global $DB;
        $this->timemodified = time();
        return $DB->update_record(static::table(), $this);
    }

    /**
     * get course module record for this file.
     */
    public function get_course_module() {
        global $DB;
        $sql = "SELECT m.* FROM {course_modules} cm JOIN {modules} m ON m.id=cm.module WHERE cm.id=?";
        return $DB->get_record_sql($sql, array($this->cm));
    }

}
