<?php
/**
 *
 * @package     mod_lanebs
 * @category    HealthCheck
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\HealthCheck;

use core_plugin_manager;
use mod_lanebs\AuthService\AuthInterface;
use mod_lanebs\Support\Client\ClientServiceInterface;
use mod_lanebs\Support\Config;
use mod_lanebs\Token\TokenInterface;

final class HealthCheck implements HealthCheckInterface
{
    public function __construct(
        protected ClientServiceInterface $client,
        protected TokenInterface         $token,
        protected AuthInterface          $auth,
    )
    {
    }

    public function getPluginVersion(): string
    {
        $allPlugins = core_plugin_manager::instance()
            ->get_present_plugins('mod');
        $plugin = $allPlugins['lanebs'];

        return $plugin->release . ' (' . $plugin->version . ')';
    }

    public function checkAllSystem(array $list = []): HealthCheckDto
    {
        $version = $this->getPluginVersion();
        return new HealthCheckDto(
            $version,
            PHP_VERSION,
            $this->getMoodleVersion(),
            $this->getOsType(),
            $this->checkHostList($list),
            $this->getToken(),
            $this->rawAuth()
        );

    }

    public function getOsType(): string
    {
        global $CFG;
        return $CFG->ostype;
    }

    public function getMoodleVersion(): string
    {
        global $CFG;
        return $CFG->release;
    }

    public function checkHostList(array $list): array
    {
        $result = [];
        foreach ($list as $key => $item) {
            $result[] = $this->checkServer($key, $item);
        }

        return $result;
    }

    protected function checkServer(string $key, string $item): array
    {
        $hasConfig = (bool)Config::get($key);
        $this->client->get($item);
        $code = $this->client->getCode();
//        $cmd = "curl -I $item | grep HTTP";
        $cmd = "curl --insecure -vvI $item 2>&1 | grep \"^HTTP\|start date:\|expire date:\"";
        exec($cmd, $output);
        return [
            $item,
            $hasConfig,
            $code,
            $output
        ];
    }

    public function getToken(): ?string
    {
        return $this->token->get();
    }

    public function rawAuth(): ?string
    {
        return $this->auth->raw();
    }
}