<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 *
 * @package     mod_lanebs
 * @category    Logger
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\Logger;

use mod_lanebs\Support\Client\ClientServiceInterface;
use mod_lanebs\Support\Config;
use mod_lanebs\Token\TokenInterface;

final class Logger implements LoggerInterface
{
    public function __construct(
        protected string                 $url,
        protected TokenInterface         $token,
        protected ClientServiceInterface $client,
        protected int                    $version
    )
    {
        $this->client->setOptional([
            'CURLOPT_TIMEOUT' => 5
        ]);
    }

    protected function send(string $path, array $param = []): string
    {
        $url = $this->url . $path;
        return (string)$this->client->post($url, $param);
    }

    public function log(array $data): string
    {
        $data['version'] = $this->version;
        $data['token'] = $this->token->get();
        $path = '/api/1.0/moodle/log?token=' . $data['token'];
        return $this->send($path, $data);
    }

    public function statistic(array $data): string
    {
        $data['version'] = $this->version;
        $data['token'] = $this->token->get();
        $path = '/api/1.0/moodle/stat?token=' . $data['token'];
        return $this->send($path, $data);
    }

    public function install(): bool
    {
        if (!$this->token->get()) {
            return false;
        }

        if (Config::get('install_log')) {
            return false;
        }

        $result = $this->statistic(['type' => 'install']);
        $result = json_decode($result);
        if (empty($result->message) || $result->message !== 'ok') {
            return false;
        }
        Config::set('install_log', time());
        return true;
    }

    public function uninstall(): bool
    {
        if (!$this->token->get()) {
            return false;
        }

        $this->statistic(['type' => 'uninstall']);
        return true;
    }
}