<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 *
 * @package     mod_lanebs
 * @category    Client
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\Support\Client;

use curl;

/** @global \stdClass $CFG */
require_once $CFG->libdir . '/filelib.php';

class BaseClientService implements ClientServiceInterface
{
    protected curl $client;

    protected array $options = [];

    public function __construct(protected ClientLoggerInterface $logger)
    {
        $this->client = new curl(['ignoresecurity' => true]);
        $this->setOptional([
            'CURLOPT_SSL_VERIFYPEER' => true,
            'CURLOPT_RETURNTRANSFER' => true,
        ]);
    }

    public function setOptional(array $options): self
    {
        $this->options += $options;
        $this->client->setopt($options);
        return $this;
    }

    public function get(string $url, array $params = [], array $options = [])
    {
        $result = $this->client->get($url, $params, $options);
        $this->logger->log(
            $result,
            $url,
            $params,
            ($this->options + $options),
            $this->client->get_info()
        );

        return $result;
    }

    public function post(string $url, array $params = [], array $options = [])
    {
        $result = $this->client->post($url, $params, $options);
        $this->logger->log(
            $result,
            $url,
            $params,
            ($this->options + $options),
            $this->client->get_info(),
        );

        return $result;
    }

    public function getCode(): ?int
    {
        $info = $this->client->get_info();
        if (!isset($info['http_code'])) {
            return null;
        }

        return (int)$info['http_code'];
    }
}