<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * lanebs of interface functions and constants.
 *
 * @package     mod_lanebs
 * @copyright   2020 Senin Yurii <katorsi@mail.ru>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/question/format/xml/format.php');
require_once($CFG->dirroot . '/user/externallib.php');
require_once($CFG->dirroot . '/lib/classes/user.php');
require_once($CFG->dirroot . '/webservice/lib.php');
require_once($CFG->libdir . '/filelib.php');


/**
 * Return if the plugin supports $feature.
 *
 * @param string $feature Constant representing the feature.
 * @return true | null True if the feature is supported, null otherwise.
 */
function lanebs_supports($feature) {
    switch ($feature) {
        case FEATURE_BACKUP_MOODLE2:
        case FEATURE_MOD_INTRO:
            return true;
        default:
            return null;
    }
}

/**
 * Saves a new instance of the mod_lanebs into the database.
 *
 * Given an object containing all the necessary data, (defined by the form
 * in mod_form.php) this function will create a new instance and return the id
 * number of the instance.
 *
 * @param object $moduleinstance An object from the form.
 * @param mod_lanebs_mod_form $mform The form.
 * @return int The id of the newly inserted record.
 */
function lanebs_add_instance($moduleinstance, $mform = null) {
    global $DB;

    $moduleinstance->timecreated = time();

    return $DB->insert_record('lanebs', $moduleinstance);
}

/**
 * Updates an instance of the mod_lanebs in the database.
 *
 * Given an object containing all the necessary data (defined in mod_form.php),
 * this function will update an existing instance with new data.
 *
 * @param object $moduleinstance An object from the form in mod_form.php.
 * @param mod_lanebs_mod_form $mform The form.
 * @return bool True if successful, false otherwise.
 */
function lanebs_update_instance($moduleinstance, $mform = null) {
    global $DB;

    $moduleinstance->timemodified = time();
    $moduleinstance->id = $moduleinstance->instance;

    return $DB->update_record('lanebs', $moduleinstance);
}

/**
 * Removes an instance of the mod_lanebs from the database.
 *
 * @param int $id Id of the module instance.
 * @return bool True if successful, false on failure.
 */
function lanebs_delete_instance($id) {
    global $DB;

    $exists = $DB->get_record('lanebs', array('id' => $id));
    if (!$exists) {
        return false;
    }

    $DB->delete_records('lanebs', array('id' => $id));

    return true;
}

/**
 * Given a course_module object, this function returns any
 * "extra" information that may be needed when printing
 * this activity in a course listing.
 *
 * See {@link get_array_of_activities()} in course/lib.php
 *
 * @param object $coursemodule
 * @return cached_cm_info info
 */
function lanebs_get_coursemodule_info($coursemodule) {
    global $DB, $PAGE;
    $code = 'require(["jquery"], function($) {$(".modtype_lanebs").find("img.activityicon").closest("div").css("background-color", "#fff");});';
    $PAGE->requires->js_amd_inline($code);
    if (!$lanebs = $DB->get_record('lanebs', array('id'=>$coursemodule->instance),
        'id, course, name, content, content_name, page_number, cover, videos, type')) {
        return NULL;
    }

    $info = new cached_cm_info();
    $info->name = $lanebs->name;
    $info->icon = null;
    if (!empty($lanebs->videos)) {
        $videos = json_decode($lanebs->videos);
        if ($videos) {
            foreach ($videos as $video) {
                $info->content .= '<p><img style="width:100px;" class="img-responsive atto_image_button_text-bottom" src="https://img.youtube.com/vi/'.$video->video_id.'/0.jpg" alt="'.$video->name.'">'.$video->name.'</p>';
            }
        }
    }

    return $info;
}

