<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Перенос функций из lib.php
 * @package   local_patterns_tools
 * @category  version
 * @copyright 2021 Mazitov Artem
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Added user, token, etc.
 *
 */
function install_requirements()
{
    global $DB, $PAGE;
    $systemContext = context_system::instance();
    $PAGE->set_context($systemContext);
    $enabledprotocols = get_config('core', 'webserviceprotocols');
    if (stripos($enabledprotocols, 'rest') === false) {
        set_config('webserviceprotocols', $enabledprotocols . ',rest');
    }
    $webserviceUser = core_user::get_user_by_email('lan@lanbook.com');
    if (!$webserviceUser) {
        $user = array(
            'username' => 'ws-lanebs-constructor',
            'firstname' => 'Lan Constructor',
            'lastname' => 'User',
            'email' => 'lan@lanbook.com',
            'createpassword' => true,
        );
        $webserviceUser = (object)core_user_external::create_users(array($user))[0];
    }
    $wsName = 'ws-lanconstructor-role';
    $wsroleId = $DB->get_record('role', array('shortname' => $wsName));
    if (!empty($wsroleId)) {
        $wsroleId = $wsroleId->id;
    } else {
        $wsroleId = create_role(get_string('lanebs_role', 'mod_lanebs'), $wsName, '');
    }
    set_role_contextlevels($wsroleId, [CONTEXT_SYSTEM]);
    assign_capability('webservice/rest:use', CAP_ALLOW, $wsroleId, $systemContext->id, true);
    assign_capability('mod/lanebs:get_constructor', CAP_ALLOW, $wsroleId, $systemContext->id, true);
    assign_capability('moodle/question:add', CAP_ALLOW, $wsroleId, $systemContext->id, true);
    assign_capability('moodle/question:managecategory', CAP_ALLOW, $wsroleId, $systemContext->id, true);
    assign_capability('moodle/question:useall', CAP_ALLOW, $wsroleId, $systemContext->id, true);
    role_assign($wsroleId, $webserviceUser->id, $systemContext->id);

    $webserviceManager = new webservice();
    $service = $webserviceManager->get_external_service_by_shortname('Constructor');
    if (!$service) {
        $service = $webserviceManager->get_external_service_by_shortname('LanConstructor');
    } else {
        $service->shortname = 'LanConstructor';
    }
    $service->enabled = true;
    $webserviceManager->update_external_service($service);
    $authUser = $webserviceManager->get_ws_authorised_user($service->id, $webserviceUser->id);
    if (!$authUser) {
        $webserviceManager->add_ws_authorised_user((object)['externalserviceid' => $service->id, 'userid' => $webserviceUser->id]);
    }
    $token = $webserviceManager->get_user_ws_tokens($webserviceUser->id);
    if (empty($token)) {
        if (function_exists('external_generate_token')) {
            $token = external_generate_token(EXTERNAL_TOKEN_PERMANENT, $service, $webserviceUser->id, $systemContext);
        }
        else {
            $token = core_external\util::generate_token(EXTERNAL_TOKEN_PERMANENT, $service, $webserviceUser->id, $systemContext);
        }
    }
    return $token;
}

function install_permissions()
{
    global $DB, $PAGE;
    $systemContext = context_system::instance();
    $PAGE->set_context($systemContext);
    $webserviceUser = core_user::get_user_by_email('lan@lanbook.com');
    if (!has_capability('moodle/question:useall', $systemContext, $webserviceUser)) {
        $wsName = 'ws-lanconstructor-role';
        $wsroleId = $DB->get_record('role', array('shortname' => $wsName));
        set_role_contextlevels($wsroleId->id, [CONTEXT_SYSTEM]);
        assign_capability('webservice/rest:use', CAP_ALLOW, $wsroleId->id, $systemContext->id, true);
        assign_capability('mod/lanebs:get_constructor', CAP_ALLOW, $wsroleId->id, $systemContext->id, true);
        assign_capability('moodle/question:add', CAP_ALLOW, $wsroleId->id, $systemContext->id, true);
        assign_capability('moodle/question:managecategory', CAP_ALLOW, $wsroleId->id, $systemContext->id, true);
        assign_capability('moodle/question:useall', CAP_ALLOW, $wsroleId->id, $systemContext->id, true);
    }
}

function send_logs($data)
{
    $version = get_lanebs_config('release');
    $token = get_config('lanebs', 'token');
    $data['version'] = $version;
    $data['token'] = $token;
    $curl = new curl();
    $options = array(
        'CURLOPT_POST'              => true,
        'CURLOPT_SSL_VERIFYPEER'    => false,
        'CURLOPT_RETURNTRANSFER'    => true,
        'CURLOPT_USERAGENT'         => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36',
        'CURLOPT_TIMEOUT'           => 5);
    $curl->setopt($options);
    $url = get_lanebs_config('base_url').'/api/1.0/moodle/log?token='.$token;
    return $curl->post($url, $data, $options);
}

function getLanebs($course, $section, $module)
{
    $lanebs = new stdClass();
    $lanebs->course = $course;
    $lanebs->timecreated = time();
    $lanebs->timemodified = time();
    $lanebs->introeditor = array('text' => '</p></p>', 'format' => 1, 'itemid' => 0);
    $lanebs->itemid = 0;
    $lanebs->visible = 1;
    $lanebs->visibleoncoursepage = 1;
    $lanebs->cmidnumber = '';
    $lanebs->availabilityconditionsjson = '{"op":"&","c":[],"showc":[]}';
    $lanebs->completionunlocked = 1;
    $lanebs->completion = 1;
    $lanebs->completionexpected = 0;
    $lanebs->coursemodule = 0;
    $lanebs->section = $section;
    $lanebs->module = $module;
    $lanebs->modulename = 'lanebs';
    $lanebs->instance = 0;
    $lanebs->add = 'lanebs';
    $lanebs->update = 0;
    $lanebs->return = 0;
    $lanebs->sr = 0;
    $lanebs->competencies = array();
    $lanebs->competency_rule = '0';
    $lanebs->submitbutton = 'Сохранить и показать';
    return $lanebs;
}

function getQuiz($course, $quizData, $quizmodule, $section)
{
    $quiz = new stdClass();
    $quiz->module = $quizmodule->id;
    $quiz->modulename = 'quiz';
    $quiz->course = $course->id;
    $quiz->name = $quizData['name'];
    $quiz->cmidnumber = 0;
    $quiz->timelimit = ($quizData['timeLimit'] * 60);
    $quiz->timeopen = 0;
    $quiz->timeclose = 0;
    $quiz->questiondecimalpoints = -1;
    $quiz->visible = 1;
    $quiz->visibleoncoursepage = 1;
    $quiz->sumgrade = $quizData['qCount'];
    $quiz->grade = $quizData['qCount'];
    $quiz->intro = $quizData['description'];
    $quiz->quizpassword = '';
    $quiz->section = $section;
    $quiz->introformat = 1;
    $quiz->preferredbehaviour = 'deferredfeedback';
    $quiz->overduehandling = 'autosubmit';
    $quiz->reviewattempt = 69888;
    $quiz->reviewcorrectness = 4352;
    $quiz->reviewmarks = 4352;
    $quiz->reviewspecificfeedback = 4352;
    $quiz->reviewgeneralfeedback = 4352;
    $quiz->reviewrightanswer = 4352;
    $quiz->reviewoverallfeedback = 4352;
    $quiz->questionsperpage = 1;
    $quiz->shuffleanswers = 1;
    $quiz->browsersecurity = '-';
    return $quiz;
}

function importQuestions($filename, $category, $course)
{
    $qformat = new qformat_xml();
    $qformat->setFilename($filename);
    //$qformat->setCatfromfile(false);
    //$qformat->setContexts($contexts);
    $qformat->setCourse($course);
    //$qformat->setContextfromfile(true);
    $qformat->setCategory($category);
    $qformat->set_display_progress(false);
    $qformat->importprocess();
    return $qformat->questionids;
}

function createFolder($name, $module, $section)
{
    $folder = new \stdClass();
    $folder->modulename = $module->name;
    $folder->module = $module->id;
    $folder->name = $name;
    $folder->intro = '<h5 style="color:#1177d1">' . $name . '</h5>';
    $folder->section = $section;
    $folder->idnumber = '';
    $folder->added = time();
    $folder->score = 0;
    $folder->indent = 0;
    $folder->visible = 1;
    $folder->visibleoncoursepage = 1;
    $folder->visibleold = 1;
    $folder->groupmode = 0;
    $folder->groupingid = 0;
    $folder->completion = 1;
    $folder->completiongradeitemnumber = '';
    $folder->completionview = 0;
    $folder->completionexpected = 0;
    $folder->showdescription = 0;
    $folder->availability = '';
    $folder->deletioninprogress = 0;
    $folder->introformat = 1;
    $folder->revision = 1;
    $folder->timemodified = time();
    $folder->display = 0;
    $folder->showexpanded = 1;
    $folder->showdownloadfolder = 1;
    $folder->files = '';
    return $folder;
}

function get_course_creator($courseid)
{
    global $DB;
    $course = $DB->get_record('course', array('id' => $courseid));
    $roles = $DB->get_records('role', array('archetype' => 'editingteacher'));
    $context = context_course::instance($courseid);
    foreach ($roles as $role) {
        $teachers = get_role_users($role->id, $context);
        if ($teachers) {
            $user = array_shift($teachers);
            if ($user) {
                $user->course_date = $course->timecreated;
                return $user;
            }
        }
    }
    // если в курсе нет учителя - берем любого админа
    $user = get_admin();
    if ($user) {
        $user->course_date = $course->timecreated;
        return $user;
    }
    return false;
}

/**
 * @param $books
 * @param $journals
 * @param $course
 * @return bool|array|object
 * @throws coding_exception
 * @throws dml_exception
 * @throws moodle_exception
 */
function create_unity_list($books, $journals, $course)
{
    global $DB;
    $listName = get_string('unity_list', 'mod_lanebs');
    $pages = $DB->get_records('page', array('course' => $course->id, 'name' => $listName));
    $newpage = true;
    $module = $DB->get_record('modules', array('name' => 'page'));
    if (!$module) {
        return array('code' => 404, 'message' => 'Module "page" not found');
    }
    foreach ($pages as $page) {
        $modulePage = $DB->get_record('course_modules', array('course' => $course->id, 'module' => $module->id, 'instance' => $page->id, 'deletioninprogress' => 0));
        if ($modulePage) {
            $newpage = false;
            break;
        }
    }
    if ($newpage) {
        $page = new \stdClass();
        $page->course = $course->id;
        $page->modulename = 'page';
        $page->module = $module->id;
        $page->visible = 1;
        $page->visibleoncoursepage = 1;
        $page->section = 0;
        $page->groupingid = 0;
        $page->name = $listName;
        $page->introformat = 1;
        $page->contentformat = 1;
        $page->display = 5;
        $page->displayoptions = '{}';
        $page->revision = 1;
        $page->content = '';
    }
    $content = $page->content;
    $elements = array();
    $date = (new \DateTime())->format('d.m.Y');
    if ($content) {
        $html = new DOMDocument();
        $contentType = '<meta http-equiv="Content-Type" content="text/html; charset=utf-8">';
        $html->loadHTML($contentType . $content);
        $elements = $html->getElementsByTagName('p');
    }
    $pCount = (count($elements) === 0) ? 1 : count($elements) + 1;
    $newElements = array();
    $biblioRecords = array();
    foreach ($books as $book) {
        $biblioRecords[] = $book['biblioRecord'];
    }
    $biblioRecords = array_unique($biblioRecords);
    foreach ($biblioRecords as $biblioRecord) {
        $hit = false;
        foreach ($elements as $element) {
            $value = substr($element->nodeValue, 0, strpos($element->nodeValue, 'дата обращения'));
            $record = substr($biblioRecord, 0, strpos($biblioRecord, 'дата обращения'));
            if (mb_strpos(html_entity_decode($value), $record) !== FALSE) {
                $hit = true;
            }
        }
        if (!$hit) {
            $record = str_replace('00.00.0000', $date, $biblioRecord);
            $record = preg_replace('@(https?://([-\w\.]+[-\w])+(:\d+)?(/([\w/_\.#-]*(\?\S+)?[^\.\s])?)?)@', '<a href="$1">$1</a>', $record); // делаем ссылки ссылками
            $newElements[] = '<p><strong>' . $pCount++ . '. </strong>' . $record . '</p>';
        }
    }
    foreach ($journals as $journal) {
        $hit = false;
        foreach ($elements as $element) {
            if (mb_strpos(html_entity_decode($element->nodeValue), $journal['biblioRecord']) !== FALSE) {
                $hit = true;
            }
        }
        if (!$hit) {
            $record = str_replace('00.00.0000', $date, $journal['biblioRecord']);
            $newElements[] = '<p><strong>' . $pCount++ . '. </strong>' . $record . '</p>';
        }
    }
    $page->content = $content . implode('', $newElements);
    $page->timemodified = time();
    $page->printintro = '0';
    $page->printlastmodified = '0';
    if ($newpage) {
        return add_moduleinfo($page, $course);
    }
    $page->revision++;
    return $DB->update_record('page', $page);
}

function sorting_section_modules($course, $section, $lanmodule, $dirmodule, $dirname)
{
    global $DB;
    $sectionModules = $DB->get_record('course_sections', array('course' => $course->id, 'section' => $section));
    $sequences = explode(',', $sectionModules->sequence);
    $seqModules = $DB->get_records_list('course_modules', 'id', $sequences);
    $newsequence = array();
    $bookids = array();
    $journalids = array();
    $videoids = array();
    $up = false;
    foreach ($seqModules as $module) {
        if ($module->module != $dirmodule->id && $module->module != $lanmodule->id) {
            $newsequence[] = $module->id;
        }
        if ($module->module == $dirmodule->id) {
            $record = $DB->get_record($dirmodule->name, array('id' => $module->instance));
            if ($record) {
                if (mb_strpos($record->name, $dirname) !== FALSE) {
                    if ($up) {
                        $newsequence = array_merge($newsequence, $bookids);
                        $newsequence = array_merge($newsequence, $journalids);
                        $newsequence = array_merge($newsequence, $videoids);
                        $bookids = array();
                        $journalids = array();
                        $videoids = array();
                        $up = false;
                    } else {
                        $up = true;
                    }
                }
                $newsequence[] = $module->id;
            }
        }
        if ($module->module == $lanmodule->id) {
            $lan = $DB->get_record('lanebs', array('id' => $module->instance));
            if ($lan->type == mod_lanebs_constructor_external::BOOK_TYPE) {
                $bookids[] = $module->id;
            } else if ($lan->type == mod_lanebs_constructor_external::JOURNAL_TYPE) {
                $journalids[] = $module->id;
            } else if ($lan->type == mod_lanebs_constructor_external::VIDEO_TYPE) {
                $videoids[] = $module->id;
            }
        }
    }
    if ($up || (!empty($bookids) || !empty($journalids) || !empty($videoids))) {
        $newsequence = array_merge($newsequence, $bookids);
        $newsequence = array_merge($newsequence, $journalids);
        $newsequence = array_merge($newsequence, $videoids);
    }
    $sectionModules->sequence = implode(',', $newsequence);
    $DB->update_record('course_sections', $sectionModules);
    return true;
}

function check_module_exist($course, $section, $moduleObj, $moduleName)
{
    global $DB;
    $courseSection = $DB->get_record('course_sections', array('course' => $course->id, 'section' => $section));
    $modules = $DB->get_records('course_modules', array('course' => $course->id, 'section' => $courseSection->id, 'module' => $moduleObj->id, 'deletioninprogress' => 0));
    foreach ($modules as $module) {
        $record = $DB->get_record($moduleObj->name, array('id' => $module->instance));
        if (mb_strpos($record->name, $moduleName) !== FALSE) {
            return true;
        }
    }
    return false;
}

function send_stat($data)
{
    $version = get_lanebs_config('release');
    $token = get_config('lanebs', 'token');
    $data['version'] = $version;
    $data['token'] = $token;
    $curl = new curl();
    $options = array(
        'CURLOPT_POST'              => true,
        'CURLOPT_SSL_VERIFYPEER'    => false,
        'CURLOPT_RETURNTRANSFER'    => true,
        'CURLOPT_USERAGENT'         => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36',
        'CURLOPT_TIMEOUT'           => 5);
    $curl->setopt($options);
    $url = get_lanebs_config('base_url').'/api/1.0/moodle/stat?token='.$token;
    return $curl->post($url, $data, $options);
}

/**
 * @deprecated
 * @param $name
 * @return mixed
 */
function get_lanebs_config($name)
{
    $plugin = new \stdClass();
    require __DIR__.'/version.php';
    return $plugin->$name;
}