<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 *
 * @package     mod_lanebs
 * @category    AuthService
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\AuthService;

use mod_lanebs\Support\Client\ClientServiceInterface;
use mod_lanebs\Token\TokenInterface;

class Auth implements AuthInterface
{
    protected ?string $authToken = null;

    public function __construct(
        protected string                 $url,
        protected TokenInterface         $token,
        protected ClientServiceInterface $client
    )
    {
        if (!empty($_SESSION['mod_lanebs_readerToken'])) {
            $this->authToken = (string)$_SESSION['mod_lanebs_readerToken'];
        }
    }

    public function raw(): ?string
    {
        $options = [
            'CURLOPT_USERAGENT' => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36',
            'CURLOPT_TIMEOUT' => 5,
            'CURLOPT_HTTPHEADER' => [
                'x-auth-token-subscriber: ' . $this->token->get()
            ]
        ];

        $result = $this->client->post($this->url, options: $options);
        return is_string($result) ? $result : null;
    }

    /**
     * @return mixed|null
     */
    public function auth(): mixed
    {
        $data = $this->raw();
        if (!$data) {
            return null;
        }

        $data = json_decode($data);
        if (empty($data->jwt->access_token)) {
            if (!empty($_SESSION['mod_lanebs_readerToken'])) {
                unset($_SESSION['mod_lanebs_readerToken']);
                $this->authToken = null;
            }
            return null;
        }

        $this->authToken = (string)$data->jwt->access_token;
        $_SESSION['mod_lanebs_readerToken'] = $this->authToken;

        return $data;
    }

    public function get(bool $force = false): ?string
    {
        if (!$this->authToken || $force) {
            $this->auth();
        }

        return $this->authToken;
    }
}