<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 *
 * @package     mod_lanebs
 * @category    Content
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\Content;

use mod_lanebs\Support\Client\ClientServiceInterface;

final class ContentService implements ContentInterface
{
    public const SORT_CREATE_DESC = 'create_time';

    public function __construct(
        protected string                 $apiUrl,
        protected ClientServiceInterface $client
    )
    {
        $this->client->setOptional([
            'CURLOPT_USERAGENT' => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36'
        ]);
    }

    protected function get(string $path, array $param = []): string
    {
        $url = $this->apiUrl . $path;
        return (string)$this->client->get($url, $param);
    }

    public function search($searchParam, int $page, int $limit, int $catId = null): string
    {
        $params = [
            'page' => $page,
            'limit' => $limit,
            'sort' => self::SORT_CREATE_DESC
        ];

        if (!empty($searchParam['eduFilter'])) {
            $params[$searchParam['eduFilter']] = 1;
        }

        if (isset($searchParam['bookFilter'])) {
            $params['type'] = $searchParam['bookFilter'];
        }

        if (!$catId) {
            $params['category_id'] = $catId;
        }

        if (!empty($searchParam['searchString'])) {
            $params['query_id_isbn_title'] = $searchParam['searchString'];
            $path = '/api/search/book';
        } else {
            $path = '/api/categories/books';
        }

        return $this->get($path, $params);
    }

    public function getCategories(): string
    {
        $path = '/api/categories';
        return $this->get($path);
    }

    public function getCategory(int $categoryId): string
    {
        $path = '/api/category/' . $categoryId;
        return $this->get($path);
    }

    public function getToc(int $id): array
    {
        $path = '/api/book/' . $id . '/seealso';
        $data = $this->get($path);
        if (!$data) {
            return [];
        }
        $data = json_decode($data);

        if (!empty($data->body->items)) {
            return [];
        }

        $result = [];
        foreach ($data->body->items as $index => $item) {
            $result[] = [
                'book_id' => (string)$item->book_id,
                'start_page' => (string)$item->start_page,
                'link_name' => $item->link_name,
                'link_url' => $item->link_url,
                'unique_id' => ++$index
            ];
        }

        return $result;
    }
}