<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 *
 * @package     mod_lanebs
 * @category    Logger
 * @copyright   2024 Mazitov Artem
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace mod_lanebs\tests\Logger;

use advanced_testcase;
use mod_lanebs\Logger\Logger;
use mod_lanebs\Logger\LoggerFactory;
use mod_lanebs\Logger\LoggerInterface;
use mod_lanebs\Support\Client\BaseClientService;
use mod_lanebs\Support\Client\ClientLoggerInterface;
use mod_lanebs\Support\Config;
use mod_lanebs\Token\TokenInterface;

final class Logger_test extends advanced_testcase
{
    protected static string $token = '';

    protected function setUp(): void
    {
        if (
            !defined('TEST_MOD_LANEBS_TOKEN') ||
            !TEST_MOD_LANEBS_TOKEN
        ) {
            $this->markTestSkipped('empty TEST_MOD_LANEBS_TOKEN');
        }

        Config::installConfig();

        $url = Config::baseUrl();
        if (!$url) {
            $this->markTestSkipped('empty api URL');
        }

        $this->resetAfterTest();
        self::$token = TEST_MOD_LANEBS_TOKEN;
        parent::setUp(); // TODO: Change the autogenerated stub
    }

    /**
     * @return iterable<array{0: TokenInterface, 1: bool}>
     */
    public static function loggerProvider(): iterable
    {
        $url = (string)Config::baseUrl();
        $token = new class implements TokenInterface {
            public ?string $token = null;

            public function get(): ?string
            {
                return $this->token;
            }
        };
        $logger = new class implements ClientLoggerInterface {
            public function log($result, string $url, array $params = [], array $options = [], $info = []): void
            {
            }
        };

        $client = new class($logger) extends BaseClientService {
            public function post(string $url, array $params = [], array $options = [])
            {
                $data = ['message' => 'ok'] + $params;
                return json_encode($data);
            }
        };
        $version = Config::version();

        yield [new Logger($url, $token, $client, $version), false];

        $token = clone $token;
        $token->token = self::$token;
        yield [new Logger($url, $token, $client, $version), true];
    }

    public function test_factory(): void
    {
        $logger = LoggerFactory::create();
        $this->assertInstanceOf(LoggerInterface::class, $logger);
    }

    /**
     * @dataProvider loggerProvider
     * @param LoggerInterface $logger
     * @param bool $hasToken
     * @return void
     */
    public function test_log(LoggerInterface $logger, bool $hasToken): void
    {
        $this->markTestSkipped('TODO add test');
        $data = [
            'type' => 'book',
        ];
        $response = $logger->log($data);
    }

    /**
     * @dataProvider loggerProvider
     * @param LoggerInterface $logger
     * @param bool $hasToken
     * @return void
     */
    public function test_statistic(LoggerInterface $logger, bool $hasToken): void
    {
        $type = 'ok';
        $response = $logger->statistic(['type' => $type]);
        $response = json_decode($response);
        $this->assertNotEmpty($response);

        $this->assertEquals('ok', $response->message);
        $this->assertEquals($type, $response->type);
        $this->assertEquals(get_config('mod_lanebs', 'version'), $response->version);
        if ($hasToken) {
            $this->assertNotEmpty($response->token);
        } else {
            $this->assertNull($response->token);
        }
    }

    /**
     * @dataProvider loggerProvider
     * @param LoggerInterface $logger
     * @param bool $hasToken
     * @return void
     */
    public function test_install(LoggerInterface $logger, bool $hasToken): void
    {
        $this->assertEquals($hasToken, $logger->install());
        if (!$hasToken) {
            return;
        }
        $this->assertFalse($logger->install());
    }

    /**
     * @dataProvider loggerProvider
     * @param LoggerInterface $logger
     * @param bool $hasToken
     * @return void
     */
    public function test_uninstall(LoggerInterface $logger, bool $hasToken): void
    {
        $this->assertEquals($hasToken, $logger->install());
    }
}