<?php
// This file is part of the tutorial booking plugin
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.
//

/**
 * Defines the web service endpoint for removing a sign up from a Tutorial booking activity.
 *
 * @package    mod_tutorialbooking
 * @author     Neill Magill <neill.magill@nottingham.ac.uk>
 * @copyright  2017 University of Nottingham
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_tutorialbooking\external;

use core_external\external_function_parameters;
use core_external\external_single_structure;
use core_external\external_value;
use mod_tutorialbooking\user;

/**
 * Defines the web service endpoint for removing a sign up from a Tutorial booking activity.
 *
 * @package    mod_tutorialbooking
 * @author     Neill Magill <neill.magill@nottingham.ac.uk>
 * @copyright  2017 University of Nottingham
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class removesignup extends \core_external\external_api {
    /**
     * Remove the users sign up from the tutorial booking the slot is in.
     *
     * @param int $slotid The id of a Tutorial booking slot.
     * @return array
     */
    public static function execute($slotid) {
        global $DB, $USER;
        $return = [];
        $session = $DB->get_record('tutorialbooking_sessions', ['id' => $slotid], '*', MUST_EXIST);
        $tutorial = $DB->get_record('tutorialbooking', ['id' => $session->tutorialid], '*', MUST_EXIST);
        list($course, $cm) = get_course_and_cm_from_instance($tutorial->id, 'tutorialbooking');
        $context = \context_module::instance($cm->id);
        // Initialise the completion.
        $completion = new \completion_info($course);
        $completion->set_module_viewed($cm);
        try {
            user::remove_user(
                    $USER->id,
                    $tutorial,
                    $completion,
                    $cm,
                    false
            );
            $return = [
                'success' => true,
                'error' => [],
            ];
        } catch (\Exception $e) {
            $return = [
                'success' => false,
                'error' => [
                    'message' => $e->getMessage(),
                ],
            ];
        }
        return $return;
    }

    /**
     * Defines the inputs for the web service method.
     *
     * @return \core_external\external_function_parameters
     */
    public static function execute_parameters() {
        return new external_function_parameters([
            'slotid' => new external_value(PARAM_INT, 'The id of a Tutorial booking slot', VALUE_REQUIRED),
        ]);
    }

    /**
     * Defines the output of the web service.
     *
     * @return \core_external\external_function_parameters
     */
    public static function execute_returns() {
        return new external_function_parameters([
            'success' => new external_value(PARAM_BOOL, 'Returns is the removal was a success', VALUE_REQUIRED),
            'error' => new external_single_structure([
                'message' => new external_value(PARAM_TEXT, 'Details about any sign up failure reason', VALUE_OPTIONAL),
            ], 'Details about any errors'),
        ]);
    }
}
