<?php
// This file is part of the Tutorial Booking activity.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Tests the tutorial booking tutorial class.
 *
 * @package     mod_tutorialbooking
 * @copyright   University of Nottingham, 2014
 * @author      Neill Magill <neill.magill@nottingham.ac.uk>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_tutorialbooking;

use stdClass;

/**
 * Tests the tutorial booking tutorial class.
 *
 * @package     mod_tutorialbooking
 * @copyright   University of Nottingham, 2014
 * @author      Neill Magill <neill.magill@nottingham.ac.uk>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @group mod_tutorialbooking
 * @group uon
 */
class tutorial_test extends \advanced_testcase {
    /**
     * Tests that tutorial::getstatsfortutorial works correctly.
     *
     * @covers \mod_tutorialbooking\tutorial::getstatsfortutorial
     */
    public function test_getstatsfortutorial() {
        $this->resetAfterTest(true);
        $generator = self::getDataGenerator()->get_plugin_generator('mod_tutorialbooking');

        $user0 = self::getDataGenerator()->create_user();
        $user1 = self::getDataGenerator()->create_user();
        $user2 = self::getDataGenerator()->create_user();
        $user3 = self::getDataGenerator()->create_user();
        $user4 = self::getDataGenerator()->create_user();
        $user5 = self::getDataGenerator()->create_user();

        $course = self::getDataGenerator()->create_course();
        $tutorialbooking = $generator->create_instance(['course' => $course->id]);
        $slot1 = $generator->add_slot($tutorialbooking, ['spaces' => 42]);
        $slot2 = $generator->add_slot($tutorialbooking, ['spaces' => 9]);
        $slot3 = $generator->add_slot($tutorialbooking, ['spaces' => 500]);
        $generator->add_slot($tutorialbooking, ['spaces' => 69]);
        $generator->signup_user($tutorialbooking, $slot1, $user0);
        $generator->signup_user($tutorialbooking, $slot1, $user1);
        $generator->signup_user($tutorialbooking, $slot2, $user2);
        $generator->signup_user($tutorialbooking, $slot3, $user3);
        $generator->signup_user($tutorialbooking, $slot2, $user4);

        $tutorialbooking2 = $generator->create_instance(['course' => $course->id]);
        $slot5 = $generator->add_slot($tutorialbooking2, ['spaces' => 5]);
        $slot6 = $generator->add_slot($tutorialbooking2, ['spaces' => 2]);
        $slot7 = $generator->add_slot($tutorialbooking2, ['spaces' => 3]);
        $generator->signup_user($tutorialbooking2, $slot5, $user0);
        $generator->signup_user($tutorialbooking2, $slot6, $user1);
        $generator->signup_user($tutorialbooking2, $slot6, $user2);
        $generator->signup_user($tutorialbooking2, $slot7, $user3);
        $generator->signup_user($tutorialbooking2, $slot7, $user4);
        $generator->signup_user($tutorialbooking2, $slot7, $user5);

        $stats = tutorial::getstatsfortutorial($tutorialbooking->id);
        $this->assertEquals(5, $stats->signedup);
        $this->assertEquals(620, $stats->places);

        $stats2 = tutorial::getstatsfortutorial($tutorialbooking2->id);
        $this->assertEquals(6, $stats2->signedup);
        $this->assertEquals(10, $stats2->places);

        $this->assertDebuggingNotCalled();
    }

    /**
     * Tests that tutorial::togglelock works correctly.
     *
     * @covers \mod_tutorialbooking\tutorial::togglelock
     */
    public function test_togglelock() {
        global $DB;
        $this->resetAfterTest(true);
        $generator = self::getDataGenerator()->get_plugin_generator('mod_tutorialbooking');

        $course = self::getDataGenerator()->create_course();
        $tutorialbooking = $generator->create_instance(['course' => $course->id, 'locked' => 0]);
        $tutorialbooking2 = $generator->create_instance(['course' => $course->id]);

        $this->assertEquals(0, $DB->get_field('tutorialbooking', 'locked', ['id' => $tutorialbooking->id]));

        $this->assertTrue(tutorial::togglelock($tutorialbooking->id, true));
        $this->assertEquals(1, $DB->get_field('tutorialbooking', 'locked', ['id' => $tutorialbooking->id]));
        $this->verify_tutorial_record_unchanged($tutorialbooking2);

        $this->assertTrue(tutorial::togglelock($tutorialbooking->id, false));
        $this->assertEquals(0, $DB->get_field('tutorialbooking', 'locked', ['id' => $tutorialbooking->id]));
        $this->verify_tutorial_record_unchanged($tutorialbooking2);

        $this->assertDebuggingNotCalled();
    }

    /**
     * Tests that tutorial::gettutorialsessions works correctly.
     *
     * @covers \mod_tutorialbooking\tutorial::gettutorialsessions
     */
    public function test_gettutorialsessions() {
        $this->resetAfterTest(true);
        $generator = self::getDataGenerator()->get_plugin_generator('mod_tutorialbooking');

        $course = self::getDataGenerator()->create_course();
        $tutorialbooking = $generator->create_instance(['course' => $course->id]);
        $slot1 = $generator->add_slot($tutorialbooking, ['sequence' => 1]);
        $slot2 = $generator->add_slot($tutorialbooking, ['sequence' => 3]);
        $slot3 = $generator->add_slot($tutorialbooking, ['sequence' => 2]);
        $slot4 = $generator->add_slot($tutorialbooking, ['sequence' => 4]);

        $tutorialbooking2 = $generator->create_instance(['course' => $course->id]);
        $slot5 = $generator->add_slot($tutorialbooking2, ['spaces' => 5]);
        $slot6 = $generator->add_slot($tutorialbooking2, ['spaces' => 2]);
        $slot7 = $generator->add_slot($tutorialbooking2, ['spaces' => 3]);

        $sessions = tutorial::gettutorialsessions($tutorialbooking->id);
        $this->assertCount(4, $sessions);
        $this->assertArrayHasKey($slot1->id, $sessions);
        $this->assertArrayHasKey($slot2->id, $sessions);
        $this->assertArrayHasKey($slot3->id, $sessions);
        $this->assertArrayHasKey($slot4->id, $sessions);
        $this->assertEquals($slot1, $sessions[$slot1->id]);
        $this->assertEquals($slot2, $sessions[$slot2->id]);
        $this->assertEquals($slot3, $sessions[$slot3->id]);
        $this->assertEquals($slot4, $sessions[$slot4->id]);

        $sessions2 = tutorial::gettutorialsessions($tutorialbooking2->id);
        $this->assertCount(3, $sessions2);
        $this->assertArrayHasKey($slot5->id, $sessions2);
        $this->assertArrayHasKey($slot6->id, $sessions2);
        $this->assertArrayHasKey($slot7->id, $sessions2);
        $this->assertEquals($slot5, $sessions2[$slot5->id]);
        $this->assertEquals($slot6, $sessions2[$slot6->id]);
        $this->assertEquals($slot7, $sessions2[$slot7->id]);

        $this->assertDebuggingNotCalled();
    }

    /**
     * Tests that tutorial::gettutorialsignups works correctly.
     *
     * @covers \mod_tutorialbooking\tutorial::gettutorialsignups
     */
    public function test_gettutorialsignups() {
        $this->resetAfterTest(true);
        $generator = self::getDataGenerator()->get_plugin_generator('mod_tutorialbooking');

        // Setup data for the test.
        $user0 = self::getDataGenerator()->create_user();
        $user1 = self::getDataGenerator()->create_user();
        $user2 = self::getDataGenerator()->create_user();
        $user3 = self::getDataGenerator()->create_user();
        $user4 = self::getDataGenerator()->create_user();
        $user5 = self::getDataGenerator()->create_user();

        $course = self::getDataGenerator()->create_course();
        $tutorialbooking = $generator->create_instance(['course' => $course->id]);
        $slot1 = $generator->add_slot($tutorialbooking, ['spaces' => 42]);
        $slot2 = $generator->add_slot($tutorialbooking, ['spaces' => 9]);
        $slot3 = $generator->add_slot($tutorialbooking, ['spaces' => 500]);
        $generator->add_slot($tutorialbooking, ['spaces' => 69]);
        $generator->signup_user($tutorialbooking, $slot1, $user0);
        $generator->signup_user($tutorialbooking, $slot1, $user1);
        $generator->signup_user($tutorialbooking, $slot2, $user2);
        $generator->signup_user($tutorialbooking, $slot3, $user3);
        $generator->signup_user($tutorialbooking, $slot2, $user4);

        $tutorialbooking2 = $generator->create_instance(['course' => $course->id]);
        $slot5 = $generator->add_slot($tutorialbooking2, ['spaces' => 5]);
        $slot6 = $generator->add_slot($tutorialbooking2, ['spaces' => 2]);
        $slot7 = $generator->add_slot($tutorialbooking2, ['spaces' => 3]);
        $generator->signup_user($tutorialbooking2, $slot5, $user0);
        $generator->signup_user($tutorialbooking2, $slot6, $user1);
        $generator->signup_user($tutorialbooking2, $slot6, $user2);
        $generator->signup_user($tutorialbooking2, $slot7, $user3);
        $generator->signup_user($tutorialbooking2, $slot7, $user4);
        $generator->signup_user($tutorialbooking2, $slot7, $user5);

        // Start the test.
        $stats = tutorial::gettutorialsignups($tutorialbooking->id);
        $this->assertCount(3, $stats); // There are no signups in slot4 so it will not be returned.
        $this->assertArrayHasKey($slot1->id, $stats);
        $this->assertArrayHasKey($slot2->id, $stats);
        $this->assertArrayHasKey($slot3->id, $stats);
        $this->assertCount(2, $stats[$slot1->id]);
        $this->assertCount(2, $stats[$slot2->id]);
        $this->assertCount(2, $stats[$slot3->id]);
        $this->assertCount(2, $stats[$slot1->id]['signedup']);
        $this->assertCount(2, $stats[$slot2->id]['signedup']);
        $this->assertCount(1, $stats[$slot3->id]['signedup']);
        $this->assertEquals(2, $stats[$slot1->id]['total']);
        $this->assertEquals(2, $stats[$slot2->id]['total']);
        $this->assertEquals(1, $stats[$slot3->id]['total']);

        $this->assertDebuggingNotCalled();
    }

    /**
     * Tests that tutorial::gettutorialstatss works correctly.
     *
     * @covers \mod_tutorialbooking\tutorial::gettutorialstats
     */
    public function test_gettutorialstats() {
        $this->resetAfterTest(true);

        $fakerecord1 = new stdClass();
        $fakerecord1->id = 1;
        $fakerecord1->tutorialid = 6;
        $fakerecord1->description = 'Test Description';
        $fakerecord1->descformat = FORMAT_PLAIN;
        $fakerecord1->spaces = 4;
        $fakerecord1->sequence = 1;
        $fakerecord1->usercreated = 2;
        $clock = $this->mock_clock_with_frozen();
        $fakerecord1->timecreated = $clock->time();
        $fakerecord1->timemodified = $fakerecord1->timecreated;

        $fakerecord2 = clone($fakerecord1);
        $fakerecord2->id = 7;
        $fakerecord2->description = 'Test 2';
        $fakerecord2->spaces = 2;
        $fakerecord2->sequence = 2;

        $fakerecord3 = clone($fakerecord1);
        $fakerecord3->id = 3;
        $fakerecord3->description = 'Test 3';
        $fakerecord3->spaces = 3;
        $fakerecord3->sequence = 3;

        // Setup data for the test.
        $stats = [
            '1' => $fakerecord1,
            '7' => $fakerecord2,
            '3' => $fakerecord3,
        ];

        $signups = [
            '1' => [
                'signedup' => [
                    ['id' => 1, 'fname' => 'Joe Bloggs'],
                    ['id' => 1, 'fname' => 'Jane Bloggs'],
                    ['id' => 1, 'fname' => 'Victoria Sax-Coberg'],
                ],
                'total' => 3,
            ],
            '7' => [
                'signedup' => [
                    ['id' => 1, 'fname' => 'Jack Beanstalk'],
                    ['id' => 1, 'fname' => 'George Greenfield'],
                ],
                'total' => 2,
            ],
            '3' => [
                'signedup' => [
                    ['id' => 1, 'fname' => 'Edith Parker'],
                ],
                'total' => 1,
            ],
        ];

        // Start the test.
        $result = tutorial::gettutorialstats($stats, $signups);
        $this->assertCount(2, $result);
        $this->assertArrayHasKey('places', $result);
        $this->assertArrayHasKey('signedup', $result);

        $this->assertEquals(9, $result['places']);
        $this->assertEquals(6, $result['signedup']);

        $this->assertDebuggingNotCalled();
    }

    /**
     * Given a tutorialbooking record, it checks that the record is still the same in the database.
     * assertEquals cannot be used as the generated tutorialbooking object it returns contains more
     * than just the tutorialbooking database table information.
     *
     * @param \stdClass $tutorialrecord The tutorialbooking record created by the generator.
     * @return void
     */
    protected function verify_tutorial_record_unchanged($tutorialrecord) {
        global $DB;
        $databaserecord = $DB->get_record('tutorialbooking', ['id' => $tutorialrecord->id]);
        $this->assertEquals($tutorialrecord->id, $databaserecord->id);
        $this->assertEquals($tutorialrecord->course, $databaserecord->course);
        $this->assertEquals($tutorialrecord->name, $databaserecord->name);
        $this->assertEquals($tutorialrecord->intro, $databaserecord->intro);
        $this->assertEquals($tutorialrecord->introformat, $databaserecord->introformat);
        $this->assertEquals($tutorialrecord->timecreated, $databaserecord->timecreated);
        $this->assertEquals($tutorialrecord->timemodified, $databaserecord->timemodified);
        $this->assertEquals($tutorialrecord->completionsignedup, $databaserecord->completionsignedup);
        $this->assertEquals($tutorialrecord->privacy, $databaserecord->privacy);
    }
}
