<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Creat invoices with ERPNext using this class
 *
 * @package local_shopping_cart
 * @author David Bogner
 * @copyright 2023 Wunderbyte GmbH
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_shopping_cart\invoice;

use core\event\base;
use core\task\manager;
use core_user;
use curl;
use local_shopping_cart\interfaces\invoice;
use local_shopping_cart\local\checkout_process\items_helper\address_operations;
use local_shopping_cart\local\vatnrchecker;
use local_shopping_cart\shopping_cart_history;
use local_shopping_cart\task\create_invoice_task;
use stdClass;

/**
 * Class erpnext_invoice. This class allows to create invoices on a remote instance of the Open Source ERP solution ERPNext.
 *
 * @author David Bogner
 * @copyright 2023 Wunderbyte GmbH
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class erpnext_invoice implements invoice {
    /**
     * @var string
     */
    private string $baseurl;
    /**
     * @var string
     */
    private string $token;
    /**
     * @var array|string[]
     */
    private array $headers;
    /**
     * @var curl curl wrapper
     */
    private curl $client;
    /**
     * @var stdClass
     */
    private stdClass $user;
    /**
     * @var string json
     */
    private string $jsoninvoice;
    /**
     * @var string json
     */
    public string $errormessage = '';
    /**
     * @var string customer name
     */
    private string $customername;
    /**
     * @var int address id from shopping_cart_address
     */
    private int $addressid = 0;
    /**
     * @var string customer company name
     */
    private string $customercompany = '';
    /**
     * @var array items on the invoice
     */
    private array $invoiceitems;
    /**
     * @var array Data structure of the invoice as array that can be json encoded.
     */
    private array $invoicedata = [];

    /**
     * @var array Payment entry from ERPNext.
     */
    private array $paymententry = [];
    /**
     * @var string Billing address.
     */
    private string $billingaddress = '';

    /**
     * Set up curl to be able to connect to ERPNext using config settings.
     */
    public function __construct() {
        global $CFG;
        // Backward compatibilty for older Moodle versions. TODO Remove in 4.5!
        require_once($CFG->dirroot . "/lib/filelib.php");

        $this->baseurl = get_config('local_shopping_cart', 'baseurl');
        $this->token = get_config('local_shopping_cart', 'token');
        $this->headers = [
                'Content-Type: application/json',
                'Authorization: token ' . $this->token,
        ];
        $this->client = new curl();
        $this->client->setHeader($this->headers);
    }

    /**
     * Create the ad hoc task for invoice creation.
     *
     * @param base $event
     * @return void
     */
    public static function create_invoice_task(base $event): void {
        $customdata = [];
        $customdata['classname'] = __CLASS__;
        $customdata['identifier'] = $event->other['identifier'];
        $createinvoicetask = new create_invoice_task();
        $createinvoicetask->set_userid($event->userid);
        $createinvoicetask->set_next_run_time(time());
        $createinvoicetask->set_custom_data($customdata);
        manager::reschedule_or_queue_adhoc_task($createinvoicetask);
    }

    /**
     * Create customer
     *
     * @param int $identifier
     * @return bool true if invoice was created, false if not
     */
    public function create_invoice(int $identifier): bool {
        global $DB;
        $url = $this->baseurl . '/api/resource/Sales Invoice';
        // Set up invoice creation.
        $this->invoiceitems = shopping_cart_history::return_data_via_identifier($identifier);

        // Set user.
        foreach ($this->invoiceitems as $item) {
            $this->addressid = (int) $item->address_billing;
            if (empty($this->user)) {
                $this->user = core_user::get_user($item->userid);
                break;
            }
            break;
        }
        // Get addressid.
        if (!$this->addressid) {
            $addressrecords = address_operations::get_all_user_addresses($this->user->id);
            if (!empty($addressrecords)) {
                $this->addressid = array_key_first($addressrecords);
            } else {
                throw new \moodle_exception(
                        'nobillingaddress',
                        'local_shopping_cart',
                        '',
                        null,
                        'No billing address available for the user.'
                );
            }
        }
        if ($this->addressid > 0 && !empty(address_operations::get_specific_user_address($this->addressid)->company)) {
            $this->customername = address_operations::get_specific_user_address($this->addressid)->company;
            $this->customercompany = $this->customername;
        } else {
            $this->customername = fullname($this->user) . ' - ' . $this->user->id;
        }

        $prepareinvoice = $this->prepare_json_invoice_data();
        if (!$prepareinvoice) {
            return false;
        }
        $customerexists = $this->customer_exists();
        if (!$customerexists) {
            if (!$this->create_customer()) {
                return false;
            }
            if (!$this->set_customer_name()) {
                return false;
            }
        }

        $response = $this->client->post(str_replace(' ', '%20', $url), $this->jsoninvoice);
        $success = $this->validate_response($response, $url);
        if ($success) {
            $invoice = new stdClass();
            $invoice->identifier = $identifier;
            $invoice->timecreated = time();
            $responsedata = json_decode($response, true);
            $invoice->invoiceid = $responsedata['data']['name'];
            $DB->insert_record('local_shopping_cart_invoices', $invoice);

            // Submit the invoice.
            $submitresponse = $this->submit_invoice($invoice->invoiceid);
            if ($submitresponse) {
                // Mark the invoice as paid.
                $paymentsuccess = $this->create_payment($responsedata['data'], $invoice->invoiceid);
                if ($paymentsuccess && $this->submit_payment_entry()) {
                    return true;
                } else {
                    mtrace("ERROR: Payment was not saved in ERPNext.");
                }
            }
        }
        return false;
    }

    /**
     * Submit invoice.
     *
     * @param string $invoiceid
     * @return bool true if invoice was submitted, false if not
     */
    public function submit_invoice(string $invoiceid): bool {
        $submiturl = $this->baseurl . '/api/resource/Sales Invoice/' . $invoiceid;
        $data = ['status' => 'Submitted', 'docstatus' => '1'];
        $submitdata = json_encode($data);
        $submitresponse = $this->client->put(str_replace(' ', '%20', $submiturl), $submitdata);
        return $this->validate_response($submitresponse, $submiturl);
    }

    /**
     * Submit payment entry in ERPNext.
     *
     * @return bool true if invoice was submitted, false if not
     */
    public function submit_payment_entry(): bool {
        $paymententryid = $this->paymententry['data']['name'];
        $submiturl = $this->baseurl . '/api/resource/Payment Entry/' . $paymententryid;
        $data = ['status' => 'Submitted', 'docstatus' => '1'];
        $submitdata = json_encode($data);
        $submitresponse = $this->client->put(str_replace(' ', '%20', $submiturl), $submitdata);
        return $this->validate_response($submitresponse, $submiturl);
    }

    /**
     * Create payment
     *
     * @param array $invoicedata
     * @param string $invoiceid
     *
     * @return bool true if invoice was submitted, false if not
     */
    public function create_payment(array $invoicedata, string $invoiceid): bool {
        $paymententryurl = $this->baseurl . '/api/resource/Payment Entry';
        $paymententrydata = json_encode([
            'payment_type' => 'Receive',
            'party_type' => 'Customer',
            'party' => $this->customername,
            'paid_amount' => $invoicedata['grand_total'],
            'received_amount' => $invoicedata['grand_total'],
            'target_exchange_rate' => 1.0,
            'paid_to' => 'Erste Bank - WB',
            'paid_to_account_currency' => 'EUR',
            'reference_no' => $invoicedata['name'] . '-' . $invoicedata['posting_date'],
            'reference_date' => date('Y-m-d'),
            'references' => [
                [
                    'reference_doctype' => 'Sales Invoice',
                    'reference_name' => $invoiceid,
                    'total_amount' => $invoicedata['grand_total'],
                    'outstanding_amount' => $invoicedata['grand_total'],
                    'allocated_amount' => $invoicedata['grand_total'],
                ],
            ],
        ]);
        $paymentresponse = $this->client->post(str_replace(' ', '%20', $paymententryurl), $paymententrydata);
        if (!empty($paymentresponse)) {
            $this->paymententry = json_decode($paymentresponse, true);
        }
        return $this->validate_response($paymentresponse, $paymententryurl);
    }

    /**
     * Get tax templates available in the ERP system.
     *
     * @return array available tax tampletes, empty if no template found.
     */
    public function get_erp_taxes_charges_templates(): array {
        // Fetch 50 templates from ERP. It should be rare to have more than 50 templates configured.
        $uncleanedurl = $this->baseurl . '/api/resource/Sales Taxes and Charges Template?limit_page_length=50';
        $url = str_replace(' ', '%20', $uncleanedurl);
        $response = $this->client->get($url);
        $success = $this->validate_response($response, $url);
        $templates = [];
        if ($success) {
            $responsearray = json_decode($response, true);
            $templates = array_column($responsearray['data'], 'name');
        } else {
            throw new \moodle_exception(
                'error',
                'local_shopping_cart',
                '',
                null,
                'There was a problem fetching tax templates from ERPNext: ' . $response
            );
        }
        return $templates;
    }

    /**
     * Set tax tamplete to use for the invoice.
     *
     * @return string tax tamplete
     */
    public function set_taxes_charges_template(): string {
        // Fetch 20 templates from ERP.
        $taxtemplates = $this->get_erp_taxes_charges_templates();

        // ToDo: This is hardcoded, for internal use only, to make tax templates generic, we have to implement additional settings.

        // Pre-Checks for finding out which template to use.
        $iseuropean = vatnrchecker::is_european($this->invoicedata['taxcountrycode'] ?? null);
        $isowncountry = vatnrchecker::is_own_country($this->invoicedata['taxcountrycode'] ?? null);
        // Condtion for EU reverse charge template.
        if ($iseuropean && !$isowncountry && in_array('EU Reverse Charge', $taxtemplates)) {
            $taxtemplate = 'EU Reverse Charge';
        } else if (!$iseuropean && in_array('Export VAT', $taxtemplates)) {
            $taxtemplate = 'Export VAT';
        } else if ($isowncountry && in_array('Austria Tax', $taxtemplates)) {
            $taxtemplate = 'Austria Tax';
        } else {
            $taxtemplate = 'Austria Tax';
        }
        return $taxtemplate;
    }

    /**
     * Get billing address of customer.
     * @return string Address of the customer or empty string
     */
    public function get_billing_address(): string {
        $addressrecord = address_operations::get_specific_user_address($this->addressid);
        if ($addressrecord) {
            // Check if the address exists in ERPNext.
            if (!empty($this->customercompany)) {
                $addresstitle = $addressrecord->company;
            } else {
                $addresstitle =
                        $addressrecord->name . ' - ' .
                        $addressrecord->city . ' - ' .
                        $addressrecord->id;
            }

            $uncleanedurl = $this->baseurl . "/api/resource/Address/" . rawurlencode($addresstitle . '-Abrechnung') . "/";
            $url = str_replace(' ', '%20', $uncleanedurl);
            $response = $this->client->get($url);
            if (!$this->validate_response($response, $url)) {
                // Create the new address.
                $response = self::create_address($addressrecord, $addresstitle);
                if (!$this->validate_response($response, $url)) {
                    throw new \moodle_exception(
                        'error',
                        'local_shopping_cart',
                        '',
                        null,
                        'There was a problem with adding the address in ERPNext: ' . $response
                    );
                }
            }
            $response = json_decode($response);
            return $response->data->name;
        } else {
            throw new \moodle_exception(
                'nobillingaddress',
                'local_shopping_cart',
                '',
                null,
                'No billing address available for the user.'
            );
        }
    }

    /**
     * Create a address on ERPNext. That is needed for invoicing.
     *
     * @param object $addressrecord
     * @param string $addresstitle
     * @return string
     */
    public function create_address(object $addressrecord, string $addresstitle): string {
        $url = $this->baseurl . '/api/resource/Address';
        $address = [];
        $address['address_title'] = $addresstitle;
        $address['address_type'] = 'Billing';
        $address['address_line1'] = $addressrecord->address;
        $address['city'] = $addressrecord->city;
        $address['pincode'] = $addressrecord->zip;
        $address['country'] = $this->get_country_name_by_code($addressrecord->state);
        $address['customer'] = $addressrecord->name;

        return $this->client->post($url, json_encode($address));
    }

    /**
     * Get ERPNext country name from country code.
     *
     * @param string $code The ISO country code (e.g. 'AT', 'DE').
     * @return string|null The country name (e.g. 'Austria'), or null if not found.
     */
    protected function get_country_name_by_code(string $code): ?string {
        $url = $this->baseurl . '/api/resource/Country?filters=[["code","=","' . $code . '"]]';
        $response = $this->client->get($url);
        if (!$this->validate_response($response, $url)) {
            throw new \moodle_exception(
                'error',
                'local_shopping_cart',
                '',
                null,
                'There was a problem with retrieving the country from ERPNext: ' . $response
            );
        }
        $data = json_decode($response);
        return $data->data[0]->name;
    }


    /**
     * Prepare the json for the REST API.
     * @return bool
     */
    public function prepare_json_invoice_data(): bool {
        $serviceperiodstart = null;
        $serviceperiodend = null;
        foreach ($this->invoiceitems as $item) {
            if (!$this->item_exists($item->itemname)) {
                return false;
            }
            if (empty($this->invoicedata['timecreated'])) {
                $this->invoicedata['timecreated'] = $item->timemodified;
            }
            $itemdata = [];
            $itemdata['item_code'] = $item->itemname;
            $itemdata['qty'] = 1;

            $this->invoicedata['taxcountrycode'] = $item->taxcountrycode;
            $this->invoicedata['vatid'] = $item->vatnumber;
            if (!isset($this->invoicedata['taxes_and_charges'])) {
                $this->invoicedata['taxes_and_charges'] = self::set_taxes_charges_template();
                if (!$this->invoicedata['taxes_and_charges']) {
                    return false;
                } else {
                    self::tax_charge_exists($this->invoicedata['taxes_and_charges']);
                }
            }
            // Always use net price to send to ERPNext. In shopping_cart_history table column price is gross.
            $itemdata['rate'] = (float) $item->price - (float) $item->tax;
            $this->invoicedata['items'][] = $itemdata;

            $itemserviceperiodstart = $item->serviceperiodstart ?? $item->timecreated;
            $itemserviceperiodend = $item->serviceperiodend ?? $item->timecreated;
            if (
                is_null($serviceperiodstart) ||
                $itemserviceperiodstart < $serviceperiodstart
            ) {
                $serviceperiodstart = $itemserviceperiodstart;
            }
            if (
                is_null($serviceperiodend) ||
                $itemserviceperiodend > $serviceperiodend
            ) {
                $serviceperiodend = $itemserviceperiodend;
            }
            $this->invoicedata['address_billing'] = $item->address_billing;
        }
        $billingaddress = $this->get_billing_address();
        if (empty($billingaddress)) {
            return false;
        }
        $this->billingaddress = $billingaddress;
        $this->invoicedata['address_billing'] = $billingaddress;
        $this->invoicedata['customer'] = $this->customername;
        $date = date('Y-m-d', $this->invoicedata['timecreated']);
        // Convert the Unix timestamp to ISO 8601 date format.
        $this->invoicedata['posting_date'] = $date;
        $this->invoicedata['set_posting_time'] = 1;
        $this->invoicedata['due_date'] = $date;
        $this->invoicedata['from'] = date('Y-m-d', $serviceperiodstart);
        $this->invoicedata['to'] = date('Y-m-d', $serviceperiodend);
        $this->invoicedata['terms'] = 'Thank you for your online payment and your trust in our services.';
        $this->invoicedata['customer_address'] = $billingaddress;
        $this->jsoninvoice = json_encode($this->invoicedata);
        return true;
    }

    /**
     * Check if the customer already exists so it is not recreated on ERPNext.
     * If we pass the same customer name again to ERPNext, a new customer with a digit attached to the
     * currently used customer is created. That is what we want to avoid.
     *
     * @return bool
     */
    public function customer_exists(): bool {
        $uncleanedurl = $this->baseurl . "/api/resource/Customer/" . rawurlencode($this->customername) . "/";
        $url = str_replace(' ', '%20', $uncleanedurl);
        $response = $this->client->get($url);
        if (!$this->validate_response($response, $url)) {
            return false;
        } else {
            $responsetaxid = json_decode($response);
            if (
                $responsetaxid->data->tax_id == '' &&
                isset($this->invoicedata['vatid']) &&
                $this->invoicedata['vatid'] !== $responsetaxid->data->tax_id
            ) {
                $responsetaxid->data->tax_id = $this->invoicedata['vatid'];
                $response = $this->client->put($url, json_encode($responsetaxid->data));
            }
            return true;
        }
    }

    /**
     * Check if the tax charge already exists so it is not recreated on ERPNext.
     *
     * @param string $taxchargestemplate
     *
     * @return bool
     */
    public function tax_charge_exists(string $taxchargestemplate): bool {
        $uncleanedurl =
            $this->baseurl . "/api/resource/Sales%20Taxes%20and%20Charges%20Template/" . rawurlencode($taxchargestemplate) . "/";
        $url = str_replace(' ', '%20', $uncleanedurl);
        $response = $this->client->get($url);
        if (!$this->validate_response($response, $url)) {
            return false;
        } else {
            $taxtemplate = json_decode($response);
            $taxes = [];
            foreach ($taxtemplate->data->taxes as $tax) {
                $taxes[] =
                    [
                        'charge_type' => $tax->charge_type,
                        'account_head' => $tax->account_head,
                        'description' => $tax->description,
                        'rate' => $tax->rate,
                    ];
            }
            $this->invoicedata['taxes'] = $taxes;
        }
        return $this->validate_response($response, $url);
    }

    /**
     * Create a customer on ERPNext. That is needed for invoicing.
     *
     * @return bool
     */
    public function create_customer(): bool {
        $url = $this->baseurl . '/api/resource/Customer';
        $customer = [];
        $customer['customer_name'] = $this->customername;
        // Todo: Hardcoded ERP values. Replace with variabls.
        if (!empty($tihs->customercompany)) {
            $customer['customer_type'] = 'Company';
        } else {
            $customer['customer_type'] = 'Individual';
        }
        $customer['customer_group'] = 'All Customer Groups';
        // Todo: Implement Customer Address.
        $countrycode = get_config('local_shopping_cart', 'defaultcountry');
        if (in_array($countrycode, $this->get_all_territories())) {
            $customer['territory'] = $countrycode;
        } else {
            // This is a value present by default in ERPNext.
            $customer['territory'] = 'All Territories';
        }
        $customer['email_id'] = $this->user->email;
        $customer['customer_details'] = "Moodle user id: " . $this->user->id;
        if (isset($this->invoicedata['vatid'])) {
            $customer['tax_id'] = $this->invoicedata['vatid'];
        }
        $response = $this->client->post($url, json_encode($customer));
        if (!$response) {
            return false;
        }

        // Now it's necessary to make sure that the connection to the address is correct.
        $data = [];
        $links = ['link_doctype' => 'Customer', 'link_name' => $this->customername];
        $data['links'] = [$links];
        $url = $this->baseurl . '/api/resource/Address/' . rawurlencode($this->billingaddress);
        $response = $this->client->put($url, json_encode($data));

        return $this->validate_response($response, $url);
    }

    /**
     * Do not use. This is dangerous and should not be done in Moodle.
     *
     * Create a tax charge on ERPNext. That is needed for invoicing.
     *
     * @return bool
     */
    public function create_tax_charge(): bool {
        $url = $this->baseurl . '/api/resource/Sales%20Taxes%20and%20Charges%20Template';
        $taxpercentage = reset($this->invoiceitems);
        $taxpercentage = $taxpercentage->taxpercentage ?? '0.0';
        $title = "Test";
        // This is the company in ERPNext which is the seller, not the customer.
        $company = $this->get_default_company();
        $taxes = [
            [
                "charge_type" => "On Net Total",
                "account_head" => "Umsatzsteuer - WB",
                "rate" => $taxpercentage,
                "description" => "VAT " . $taxpercentage . "%",
            ],
        ];
        $taxtemplate = [
            "title" => $title,
            "company" => $company,
            "taxes" => $taxes,
        ];
        $response = $this->client->post($url, json_encode($taxtemplate));
        if (!$response) {
            return false;
        }
        return $this->validate_response($response, $url);
    }

    /**
     * Check if the item exists on ERPNext. If not, it is not possible to create an invoice.
     * TODO: Create item if it does not exist.
     *
     * @param string $itemname
     * @return bool
     */
    public function item_exists(string $itemname): bool {
        $url = $this->baseurl . '/api/resource/Item/' . $itemname . "/";
        $response = $this->client->get(str_replace(' ', '%20', $url));
        if (!$response) {
            return false;
        }
        return $this->validate_response($response, $url);
    }

    /**
     * Check if entry exists in the JSON response.
     *
     * @param string $response The JSON response from ERPNext.
     * @param string $url of the request response from ERPNext.
     * @return bool True if the entry exists, false otherwise.
     */
    public function validate_response(string $response, string $url): bool {
        // Decode the JSON response into an associative array.
        $resparray = json_decode($response, true);
        // Check if the response contains data.
        if (isset($resparray['data']) || isset($resparray['message'])) {
            return true; // Entry exists or entry was successfully created.
        }
        // Check if the response contains an error message.
        if (isset($resparray['exc_type'])) {
            $this->errormessage = $resparray['exc_type'] . ' - ' . $url;
            return false; // Entry does not exist (error).
        }
        if (isset($resparray['exception'])) {
            $this->errormessage = $resparray['exception'] . ' - ' . $url;
            return false; // Entry does not exist (error).
        }
        return false;
    }

    /**
     * Get all territories from ERP so we can check if they match the value used in Moodle.
     * Empty array is returned if request had a problem.
     *
     * @return string[] Array of territory names (countries and regions like EU)
     */
    private function get_all_territories(): array {
        $url = $this->baseurl . '/api/resource/Territory/';
        $response = $this->client->get($url);
        if (!$response) {
            return [];
        }
        $success = $this->validate_response($response, $url);
        if ($success) {
            $territoryarray = json_decode($response, true);
            if (isset($territoryarray['data'])) {
                return array_column($territoryarray['data'], 'name');
            }
        }
        return [];
    }

    /**
     * Set customer name, as it is not set correctly during customer creation.
     *
     * @return bool
     */
    private function set_customer_name(): bool {
        $url = $this->baseurl . '/api/resource/Customer/' . rawurlencode($this->customername);
        if (!empty($this->customercompany)) {
            $customer = ['customer_name' => $this->customername];
        } else {
            $customer = ['customer_name' => fullname($this->user)];
        }
        $json = json_encode($customer);
        $response = $this->client->put(str_replace(' ', '%20', $url), $json);
        if (!$response) {
            return false;
        }
        return $this->validate_response($response, $url);
    }

    /**
     * Get the default company name from ERPNext.
     *
     * @return string Returns the default company name or empty string if not found.
     */
    public function get_default_company(): string {
        // API endpoint to get the default company.
        $url = $this->baseurl . '/api/resource/Company';

        // Make GET request to ERPNext API.
        $response = $this->client->get($url);

        $success = $this->validate_response($response, $url);
        // If the response is not valid, return null.
        if (!$success) {
            return '';
        }

        // Decode the response body.
        $data = json_decode($response, true);

        // Validate the response structure and ensure data exists.
        if (!isset($data['data'][0]['name'])) {
            return '';
        }

        // Return the default company name.
        return $data['data'][0]['name'];
    }
}
