<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_booking;

use advanced_testcase;
use mod_booking\singleton_service;
use mod_booking\booking_bookit;
use mod_booking\bo_availability\bo_info;
use tool_mocktesttime\time_mock;
use stdClass;
use mod_booking_generator;

/**
 * Tests for booking_answers class.
 * This test checks if getters are returning the expected values.
 * Getter functions : get_usersonlist, get_usersonwaitinglist, get_usersreserved.
 *
 * @package mod_booking
 * @copyright 2022 Wunderbyte GmbH <info@wunderbyte.at>
 * @author 2025 Mahdi Poustini
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 */
final class booking_answers_test extends advanced_testcase {
    /**
     * Tests set up.
     */
    public function setUp(): void {
        parent::setUp();
        $this->resetAfterTest(true);
        time_mock::init();
        time_mock::set_mock_time(strtotime('now'));
        singleton_service::destroy_instance();
    }

    /**
     * Mandatory clean-up after each test.
     */
    public function tearDown(): void {
        parent::tearDown();
        /** @var mod_booking_generator $plugingenerator */
        $plugingenerator = self::getDataGenerator()->get_plugin_generator('mod_booking');
        $plugingenerator->teardown();
    }

    /**
     * Setup environment.
     * @return array
     */
    private function setup_booking_environment(): array {
        global $DB;

        $this->setAdminUser();

        set_config('timezone', 'Europe/Kyiv');
        set_config('forcetimezone', 'Europe/Kyiv');

        time_mock::init();
        time_mock::set_mock_time(strtotime('now'));

        // Setup test data.

        $student1 = $this->getDataGenerator()->create_user();
        $student2 = $this->getDataGenerator()->create_user();
        $student3 = $this->getDataGenerator()->create_user();
        $student4 = $this->getDataGenerator()->create_user();
        $student5 = $this->getDataGenerator()->create_user();
        $student6 = $this->getDataGenerator()->create_user();
        $student7 = $this->getDataGenerator()->create_user();

        // Create courses.
        $course = $this->getDataGenerator()->create_course(['enablecompletion' => 1]);

        $this->getDataGenerator()->enrol_user($student1->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student2->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student3->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student4->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student5->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student6->id, $course->id, 'student');
        $this->getDataGenerator()->enrol_user($student7->id, $course->id, 'student');

        // Create a booking module.
        $booking = $this->getDataGenerator()->create_module('booking', [
            'name' => 'Booking module 1',
            'course' => $course->id,
            'cancancelbook' => 0,
        ]);

        return [
            'course' => $course,
            'bookingmodule' => $booking,
            'users' => [
                'student1' => $student1,
                'student2' => $student2,
                'student3' => $student3,
                'student4' => $student4,
                'student5' => $student5,
                'student6' => $student6,
                'student7' => $student7,
            ],
        ];
    }

    /**
     *  Tests booking answer class.
     * @param array $optionsettings
     * @param array $expected
     * @return void
     *
     * @dataProvider booking_answers_all_methods_dataprovider
     * @covers \mod_booking\booking_answers\booking_answers
     */
    public function test_booking_answers_all_methods(array $optionsettings, array $expected): void {
        global $DB;
        $this->resetAfterTest(true);
        // Initial config.
        $env = $this->setup_booking_environment();
        $course = $env['course'];
        $bookingmodule = $env['bookingmodule'];
        $student1 = $env['users']['student1'];
        $student2 = $env['users']['student2'];
        $student3 = $env['users']['student3'];
        $student4 = $env['users']['student4'];
        $student5 = $env['users']['student5'];
        $student6 = $env['users']['student6'];
        $student7 = $env['users']['student7'];

        /** @var mod_booking_generator $plugingenerator */
        $plugingenerator = self::getDataGenerator()->get_plugin_generator('mod_booking');

        // Create booking option 1. Without price. No confirmation.
        $record = new stdClass();
        $record->bookingid = $bookingmodule->id;
        $record->courseid = $course->id;
        $record->text = 'Test option1';
        $record->chooseorcreatecourse = 1; // Reqiured.
        $record->maxanswers = 3;
        $record->maxoverbooking = 3;
        $record->waitforconfirmation = 0;
        foreach ($optionsettings as $itemkey => $itemvalue) {
            $record->{$itemkey} = $itemvalue;
        }
        $option1 = $plugingenerator->create_option($record);
        $settings1 = singleton_service::get_instance_of_booking_option_settings($option1->id);
        $boinfo1 = new bo_info($settings1);

        // We start booking options for users. And check if everything is as expected.

        // Book option 1 for student 1.
        $this->setUser($student1);
        booking_bookit::bookit('option', $settings1->id, $student1->id); // Try to book.
        booking_bookit::bookit('option', $settings1->id, $student1->id); // Confirm booking.
        [$id, $isavailable, $description] = $boinfo1->is_available($settings1->id, $student1->id, true);
        $this->assertEquals(MOD_BOOKING_BO_COND_ALREADYBOOKED, $id); // Check if booked.
        $this->setAdminUser();
        $bookinganswers1 = singleton_service::get_instance_of_booking_answers($settings1);
        $this->assertCount(1, $bookinganswers1->get_users());  // Total of booked users & users on waiting list.
        $this->assertCount(1, $bookinganswers1->get_usersonlist());  // Total of booked users.
        $this->assertCount(0, $bookinganswers1->get_usersonwaitinglist());  // Total users on waiting list.
        $this->assertCount(0, $bookinganswers1->get_usersdeleted());  // Total of deleted users.
        // Book option 1 for student 2 & 3.
        $this->setUser($student2);
        booking_bookit::bookit('option', $settings1->id, $student2->id);
        booking_bookit::bookit('option', $settings1->id, $student2->id);
        $this->setUser($student3);
        booking_bookit::bookit('option', $settings1->id, $student3->id);
        booking_bookit::bookit('option', $settings1->id, $student3->id);
        $this->setAdminUser();
        $bookinganswers1 = singleton_service::get_instance_of_booking_answers($settings1);
        $this->assertCount(3, $bookinganswers1->get_users());  // Total of booked & on waiting list users.

        // Book option 1 for student 4, 5 & 6. Must be on waiting list.
        $this->setUser($student4);
        booking_bookit::bookit('option', $settings1->id, $student4->id);
        booking_bookit::bookit('option', $settings1->id, $student4->id);
        $this->setUser($student5);
        booking_bookit::bookit('option', $settings1->id, $student5->id);
        booking_bookit::bookit('option', $settings1->id, $student5->id);
        $this->setUser($student6);
        booking_bookit::bookit('option', $settings1->id, $student6->id);
        booking_bookit::bookit('option', $settings1->id, $student6->id);
        $this->setAdminUser();
        $bookinganswers1 = singleton_service::get_instance_of_booking_answers($settings1);
        $this->assertCount(6, $bookinganswers1->get_users());
        $this->assertCount(3, $bookinganswers1->get_usersonlist());
        $this->assertCount(3, $bookinganswers1->get_usersonwaitinglist());
        $this->assertCount(0, $bookinganswers1->get_usersdeleted());
        $this->setUser($student7);
        // More students sould see fully booked as no more free palces is available.
        [$id, $isavailable, $description] = $boinfo1->is_available($settings1->id, $student7->id, true);
        $this->assertEquals(MOD_BOOKING_BO_COND_FULLYBOOKED, $id); // Check if booked.

        $this->setAdminUser();
        // Student 1 cancels. In this case:
        // - We souldn't see more in booked users but should see in deleted ones.
        // - Number of booked should be eqaul to 5.
        // - The student 4 on the waiting list must be moved to booked users.
        // - The persons on waiting list should be equal to 2.
        singleton_service::destroy_booking_option_singleton($option1->id);
        $option1 = singleton_service::get_instance_of_booking_option($settings1->cmid, $settings1->id);
        $option1->delete_responses([$student1->id]);
        $bookinganswers1 = singleton_service::get_instance_of_booking_answers($settings1);

        // Check multiple bookings if the opton is enabled.
        if (!empty($optionsettings['multiplebookings'])) {
            // We advance the time and check bo_availabilty to see if expectation are met.
            $time = time_mock::get_mock_time();
            $this->assertSame(time(), $time);
            $clockforwardshift = $time + $optionsettings['allowtobookagainafter'] + 20;
            time_mock::set_mock_time($time + $clockforwardshift); // Jump N seconds into the future.
            $future = time_mock::get_mock_time();
            $this->assertEquals(time(), $future);

            // Book again option for students 2 and 3.
            $this->setUser($student2);
            booking_bookit::bookit('option', $settings1->id, $student2->id);
            booking_bookit::bookit('option', $settings1->id, $student2->id);
            $this->setUser($student3);
            booking_bookit::bookit('option', $settings1->id, $student3->id);
            booking_bookit::bookit('option', $settings1->id, $student3->id);
            $a = $DB->get_records('booking_answers', ['waitinglist' => 6]);

            // We advance the time and check bo_availabilty to see if expectation are met.
            $time = time_mock::get_mock_time();
            $this->assertSame(time(), $time);
            $clockforwardshift = $time + $optionsettings['allowtobookagainafter'] + 30;
            time_mock::set_mock_time($time + $clockforwardshift); // Jump N seconds into the future.
            $future = time_mock::get_mock_time();
            $this->assertEquals(time(), $future);

            // Book again option for students 2 and 3.
            $this->setUser($student2);
            booking_bookit::bookit('option', $settings1->id, $student2->id);
            booking_bookit::bookit('option', $settings1->id, $student2->id);
            $this->setUser($student3);
            booking_bookit::bookit('option', $settings1->id, $student3->id);
            booking_bookit::bookit('option', $settings1->id, $student3->id);

            $previouslybooked = $bookinganswers1->get_userspreviouslybooked();
            $this->assertCount($expected['count_previouslybooked'][1], $previouslybooked[$student3->id]);
            // TODO: MDL-0 When Student 2 books the option again, it should not be placed on the waiting list,
            // since Student 2 has previously booked this option. Currently, the student goes on the waiting list,
            // which is not the expected behavior. Commented out for now to address later
            // Commented. $this->assertCount($expected['count_previouslybooked'][1], $previouslybooked[$student2->id]);.
        }

        $this->setAdminUser();
        $this->assertCount($expected['count_answers'], $bookinganswers1->get_answers());
        $this->assertCount($expected['count_users'], $bookinganswers1->get_users());
        $this->assertCount($expected['count_usersonlist'], $bookinganswers1->get_usersonlist());
        $this->assertCount($expected['count_usersonwaitinglist'], $bookinganswers1->get_usersonwaitinglist());
        $this->assertCount($expected['count_previouslybooked'][0], $bookinganswers1->get_userspreviouslybooked());

        // Very basic verification if pricecategory (default, no specific) is correctly added to tables.
        // Verify the pricecategory directly in the table, as it is currently not needed in booking_answers objects.
        $answers = $DB->get_records('booking_answers', ['optionid' => $option1->id]);
        foreach ($answers as $a) {
            $this->assertSame('default', $a->pricecategory);
        }
        $historyitems = $DB->get_records('booking_history');
        foreach ($historyitems as $item) {
            if (
                $item->status === MOD_BOOKING_STATUSPARAM_BOOKED
                || $item->status === MOD_BOOKING_STATUSPARAM_WAITINGLIST
            ) {
                $this->assertStringContainsString('default', $item->json);
            } else if ($item->status === MOD_BOOKING_STATUSPARAM_DELETED) {
                $this->assertEmpty($item->json);
            }
        }
    }

    /**
     * Dataprovider.
     * @return array
     */
    public static function booking_answers_all_methods_dataprovider(): array {
        return [
            'Option 1 - Multiplebookings disabled' => [
                'optionsettings' => [],
                'expected' => [
                    'count_answers' => 5,
                    'count_users' => 5,
                    'count_usersonlist' => 3,
                    'count_usersonwaitinglist' => 2,
                    'count_previouslybooked' => [0, 0],
                ],
            ],
            'Option 2 - Multiplebookings enabled' => [
                'optionsettings' => [
                    'multiplebookings' => 1, // Allow to book again.
                    'allowtobookagainafter' => 60, // Allow to book again after 60 seconds.
                ],
                'expected' => [
                    'count_answers' => 5,
                    'count_users' => 5,
                    'count_usersonlist' => 3,
                    'count_usersonwaitinglist' => 2,
                    'count_previouslybooked' => [2, 2],
                ],
            ],
        ];
    }
}
